/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.pipeline;

import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;

import lombok.Getter;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.relation.PeekingTupleIterator;
import nz.org.riskscape.problem.ResultOrProblems;

/**
 * Accepts tuples, while providing a way to iterate over the tuples that were
 * received.
 */
public class TupleIteratorSink implements Sink {

  public static final SinkConstructor CONSTRUCTOR = eo -> ResultOrProblems.of(new TupleIteratorSink());

  /**
   * Iterator that reads tuples as they are accepted by the sink
   */
  public class Iterator extends PeekingTupleIterator {

    private final ArrayBlockingQueue<Tuple> buffer = new ArrayBlockingQueue<>(100);

    private AtomicBoolean exhausted = new AtomicBoolean(false);

    @Override
    protected Tuple get() {

      while (true) {
        Tuple got;
        try {
          got = buffer.poll(100, TimeUnit.MILLISECONDS);
        } catch (InterruptedException e) {
          throw new RuntimeException("unexpected interrupt");
        }

        if (got != null) {
          return got;
        } else if (exhausted.get()) {
          // we're done
          return null;
        }

        // got nothing yet - loop and try again
      }
    }

    @Override
    protected String getSource() {
      return "unknown";
    }

    public void finish() {
      exhausted.set(true);
    }
  }

  @Getter
  private final Iterator instance = new Iterator();

  @Override
  public boolean accept(Tuple tuple) {
    return instance.buffer.offer(tuple);
  }

  @Override
  public void finish() {
    instance.finish();
  }
}
