/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.pipeline;

import java.util.HashSet;
import java.util.IdentityHashMap;
import java.util.Iterator;
import java.util.Set;
import java.util.function.Function;

import lombok.RequiredArgsConstructor;
import nz.org.riskscape.pipeline.StepNamingPolicy;
import nz.org.riskscape.pipeline.ast.PipelineDeclaration;
import nz.org.riskscape.pipeline.ast.StepDefinition;

/**
 * Assigns a name to steps based on their step id and order in the AST
 */
@RequiredArgsConstructor
public class DefaultStepNamingPolicy implements StepNamingPolicy {

  public static final DefaultStepNamingPolicy INSTANCE = new DefaultStepNamingPolicy();

  /**
   * A set of step names that have already been used - the step name function won't return these.
   */
  private final Set<String> taken;

  public DefaultStepNamingPolicy() {
    taken = Set.of();
  }

  @Override
  public Function<StepDefinition, String> getStepNameFunction(PipelineDeclaration decl) {
    // we use an identity hash map, otherwise identical, but individual, steps would get the same name
    IdentityHashMap<StepDefinition, String> map = new IdentityHashMap<>();

    // names are tracked in a standard set - the map's value collection uses identity instead of equals
    Set<String> names = new HashSet<>(taken);

    // chuck all the assigned names in the set to avoid giving an implicit name that the user might have allocated after
    // in ast order
    decl.stepDefinitionIterator().forEachRemaining(sd -> sd.getName().ifPresent(name -> names.add(name)));

    Iterator<StepDefinition> iter = decl.stepDefinitionIterator();
    while (iter.hasNext()) {
      StepDefinition step = iter.next();
      String name = step.getName().orElse(null);

      if (name == null) {
        name = step.getIdent();
        int index = 0;

        // keep looking for an unused name
        while (names.contains(name)) {
          name = step.getIdent() + "_" + ++index;
        }
      }

      names.add(name);
      map.put(step, name);
    }

    // function.apply just wraps the map lookup
    return sd -> map.get(sd);
  }
}
