/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.pipeline;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;

import com.codahale.metrics.MetricRegistry;

import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;
import nz.org.riskscape.engine.Engine;
import nz.org.riskscape.engine.Project;
import nz.org.riskscape.engine.bind.BindingContext;
import nz.org.riskscape.engine.bind.DefaultBindingContext;
import nz.org.riskscape.engine.rl.ExpressionRealizer;
import nz.org.riskscape.engine.rl.RealizationContext;
import nz.org.riskscape.engine.util.FileUtils;
import nz.org.riskscape.engine.util.ParentingListener;

@RequiredArgsConstructor
@Slf4j
public abstract class DefaultPipelineExecutor implements PipelineExecutor {

  protected final Engine engine;

  private static final int DEFAULT_SINGLE_THREADED = 1;

  @Getter @Setter
  private int numThreads = DEFAULT_SINGLE_THREADED;

  @RequiredArgsConstructor
  private class Ctx implements ExecutionContext {

    @Getter
    private final Project project;
    @Getter
    private final Engine engine;
    @Getter
    private final Path tempDirectory;

    Ctx(Project project) {
      this.project = project;
      this.engine = project.getEngine();
      try {
        this.tempDirectory = Files.createTempDirectory(engine.getTempDirectory(), "execution-context-");
      } catch (IOException e) {
        throw new RuntimeException(e);
      }
      RealizationContext realizationContext = project.newRealizationContext();
      this.bindingContext = new DefaultBindingContext(project, realizationContext, tempDirectory);
      this.expressionRealizer = realizationContext.getExpressionRealizer();
    }

    @Getter
    private final MetricRegistry metricRegistry = new MetricRegistry();

    @Getter
    private final BindingContext bindingContext;

    @Getter
    private final ExpressionRealizer expressionRealizer;

    @Override
    public PipelineExecutor getPipelineExecutor() {
      return DefaultPipelineExecutor.this;
    }

    @Override
    public ExecutionContext createNestedContext(String name) {
      Ctx child = new Ctx(project);
      child.metricRegistry.addListener(new ParentingListener(this.metricRegistry, name + "."));
      return child;
    }

    @Override
    public void close() {
      try {
        if (Files.exists(tempDirectory)) {
          FileUtils.removeDirectory(tempDirectory);
        }
      } catch (IOException e) {
        log.warn("could not delete execution context temp directory", e);
      }
    }

    @Override
    public PipelineRealizer getPipelineRealizer() {
      return new DefaultPipelineRealizer();
    }
  }

  @Override
  public ExecutionContext newExecutionContext(Project project) {
    return new Ctx(project);
  }


  public boolean isParallelizationEnabled() {
    return numThreads > DEFAULT_SINGLE_THREADED;
  }

}
