/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */

package nz.org.riskscape.engine.pipeline;

import java.util.ArrayList;
import java.util.List;
import java.util.function.Function;
import java.util.stream.Collectors;

import nz.org.riskscape.engine.bind.BoundParameters;
import nz.org.riskscape.engine.bind.ParamProblems;
import nz.org.riskscape.engine.bind.Parameter;
import nz.org.riskscape.engine.bind.ParameterSet;
import nz.org.riskscape.engine.data.BaseBookmarkResolver;
import nz.org.riskscape.engine.model.ModelFramework;
import nz.org.riskscape.engine.problem.GeneralProblems;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.engine.model.Model;

/**
 * Base class for {@link ModelFramework}s - so far this is just a place for a convenient method for handling an
 * incomplete {@link ParameterSet} to live.
 *
 * TOOD consider making this a bit more structured and follow an open-closed design, like with
 * {@link BaseBookmarkResolver}
 */
public abstract class BaseModelFramework implements ModelFramework {

  private List<Problem> ignoreIncompleteParameters(BoundParameters bound) {
    List<Problem> invalidParms = new ArrayList<>();

    // for templated models, the fact that a required parameter hasn't been specified
    // yet isn't actually a problem until we go to run the model
    Problem toIgnore = GeneralProblems.get().required("");
    for (Problem problem : bound.getValidationProblems()) {
      if (!toIgnore.getCode().equals(problem.getCode())) {
        invalidParms.add(problem);
      }
    }
    return invalidParms;
  }

  /**
   * Builds the model if the supplied parameters have no problems except for missing parameters. We
   * expect some parameters to be missing, they are for the user to enter when running the model.
   *
   * Note that warnings will be added if there are any unexpected parameters.
   *
   * @param modelParameters
   * @param modelBuilder
   * @return ResultOrProblems<Model>
   */
  protected ResultOrProblems<Model> buildWithIncompleteParameters(BoundParameters modelParameters,
      Function<BoundParameters, Model> modelBuilder) {
    // check that the specified model parameters are all valid at this point
    List<Problem> problems = ignoreIncompleteParameters(modelParameters);

    if (! modelParameters.getExtraneous().isEmpty()) {
      // there are extra parameters, maybe they are misspelt.
      problems.add(ParamProblems.get().ignoredWithHints(
          modelParameters.getExtraneous().keySet(),
          // add hints for what parameters are available
          modelParameters.getBoundTo().getDeclared().stream()
              .map(Parameter::getName)
              .collect(Collectors.toSet())
      ));
    }

    if (! Problem.hasErrors(problems)) {
      // model is either valid or not complete yet
      return ResultOrProblems.of(modelBuilder.apply(modelParameters))
          .withMoreProblems(problems);
    } else {
      // model can't be run due to invalid parameters
      return ResultOrProblems.failed(problems);
    }
  }

}
