/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */

package nz.org.riskscape.engine.output;

import java.util.Date;
import java.util.Set;

import org.locationtech.jts.geom.CoordinateSequence;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.GeometryFactory;

import com.google.common.collect.Sets;

import nz.org.riskscape.engine.types.Nullable;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.types.Types;

/**
 * The Shapefile format does not support null values. Some shapefile readers such as ogr appear to
 * handle nulls. ESRI software maps a null value to placeholder values on writing.
 */
public class ShapeFileNullMapper {

  public static final String NULL_VALUE_TEXT = "";
  public static final Number NULL_VALUE_NUMBER = -9999L;
  public static final Boolean NULL_VALUE_BOOLEAN = Boolean.FALSE;
  public static final Date NULL_VALUE_DATE = new Date(0); //epoch seems a good null value
  public static final Geometry NULL_VALUE_GEOMETRY;
  static {
    GeometryFactory gf = new GeometryFactory();
    NULL_VALUE_GEOMETRY = gf.createPoint((CoordinateSequence)null);
  }

  /**
   * Types that will be converted to {@link #NULL_VALUE_NUMBER} should their value be null.
   */
  private static final Set<Type> NULL_NUMBER_TYPES = Sets.newHashSet(
      Types.INTEGER, Types.FLOATING, Types.DECIMAL);

  /**
   * Types that will be converted to {@link #NULL_VALUE_TEXT} should their value be null.
   *
   * Geometry is in this list as they are written as text for all but the first GEOM which itself
   * should not be null.
   */
  private static final Set<Type> NULL_TEXT_TYPES = Sets.newHashSet(
      Types.TEXT, Types.ANYTHING, Types.GEOMETRY);

  /**
   * Returns the value, or a mapped value should it be null. The mapped value is safe to be used
   * in shapefiles instead of the actual null value.
   *
   * @param value to map
   * @param type the {@link Type} that value should be
   * @return value or a mapped value if it is null and a mapping is known
   */
  public Object mapValueIfNecessary(Object value, Type type) {
    if (value != null) {
      return value;
    }
    Type baseType = Nullable.unwrap(type);
    if (baseType.equals(Types.GEOMETRY)) {
      return NULL_VALUE_GEOMETRY;

    } else if (NULL_TEXT_TYPES.contains(baseType)) {
      return NULL_VALUE_TEXT;

    } else if (NULL_NUMBER_TYPES.contains(baseType)) {
      return NULL_VALUE_NUMBER;

    } else if (baseType.equals(Types.BOOLEAN)) {
      return Boolean.FALSE;

    } else if (baseType.equals(Types.DATE)) {
      return NULL_VALUE_DATE;

    }
    return null;
  }

}
