/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.output;

import java.io.IOException;
import java.io.OutputStreamWriter;
import java.net.URI;

import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.resource.CreateHandle;
import nz.org.riskscape.engine.types.Struct.StructMember;

import lombok.Getter;

/**
 * Produces a stream of json formatted lines representing each tuple.  Not to be confused with GeoJSON, which is a
 * different thing
 */
public class JSONWriter extends RiskscapeWriter {

  private final CreateHandle handle;
  private final OutputStreamWriter writer;

  @Getter
  private URI storedAt = null;


  public JSONWriter(CreateHandle handle) {
    this.handle = handle;
    this.writer = new OutputStreamWriter(handle.getOutputStream());
  }

  @Override
  public void close() throws IOException {
    writer.close();
    this.storedAt = handle.store();
  }

  @Override
  public void write(Tuple value) {
    // need a new one for each tuple, as each line is a separate json object
    org.json.JSONWriter jsonWriter = new org.json.JSONWriter(this.writer);
    writeInternal(value, jsonWriter);
    try {
      this.writer.append('\n');
      this.writer.flush();
    } catch (IOException e) {
      throw new RuntimeException(e);
    }
  }

  protected void writeInternal(Tuple value, org.json.JSONWriter jsonWriter) {

    jsonWriter.object();

    for (StructMember entry : value.getStruct().getMembers()) {
      jsonWriter.key(entry.getKey());

      Object childValue = value.fetch(entry);
      if (childValue instanceof Tuple) {
        writeInternal((Tuple) childValue, jsonWriter);
      } else {
        jsonWriter.value(childValue);
      }
    }

    jsonWriter.endObject();

  }

}
