/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */

package nz.org.riskscape.engine.output;

import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.util.Optional;

import nz.org.riskscape.engine.pipeline.ExecutionContext;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.Problems;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.problem.StandardCodes;

public abstract class BaseShapefileFormat extends BaseFormat {

  public BaseShapefileFormat(String id, String extension, String mediaType) {
    super(id, extension, mediaType, Characteristics.MULTIPLE_FILES);
  }

  @Override
  public Optional<WriterConstructor> getWriterConstructor() {
    return Optional.of((context, type, handle, unusedOptions) -> {
      // shapefile doesn't work nicely with streams, because it's not a single file (although we could support that
      // ish via zip). So we just close the handle we got given (creating an empty file) and overwrite it
      // (Note we have only checked for the presence of the .shp here - in the corner-case that
      // remnant .shx/.dbf/etc files were still present, then we will just overwrite them)
      URI outputURI = handle.store();
      File saveTo = new File(outputURI);

      if (!containsGeometry(type)) {
        // There is no geometry member. You can't have a spatial file with no geometry
        return ResultOrProblems.failed(Problem.error(StandardCodes.GEOMETRY_REQUIRED, type));
      }
      try {
        return ResultOrProblems.of(getWriter(type, saveTo, context));
      } catch (IOException e) {
        return ResultOrProblems.failed(Problems.caught(e));
      }
    });
  }

  protected abstract RiskscapeWriter getWriter(Struct type, File saveTo, ExecutionContext context) throws IOException;

}
