/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.join;

import java.util.Optional;

import lombok.Getter;

import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.relation.TupleIterator;
import nz.org.riskscape.engine.rl.ExpressionRealizer;

/**
 * Base class that collaborates with the {@link DefaultJoiner} to provide the rhs of a join, given the lhs.
 */
public abstract class JoinIndexer {

  /**
   * Cheap way of avoiding a full-on factory pattern thing - each Indexer implementation should offer a Constructor to
   * build an indexer from the given arguments.  Seemed cleaner and is compile safe), compared to introspection.
   */
  public interface Constructor {
    JoinIndexer construct(Join join, ExpressionRealizer expressionRealizer, Optional<Integer> initialIndexSize);
  }

  @Getter
  protected final Join join;

  @Getter
  protected final ExpressionRealizer expressionRealizer;

  public JoinIndexer(Join join, ExpressionRealizer expressionRealizer) {
    this.join = join;
    this.expressionRealizer = expressionRealizer;
  }

  public abstract void addToIndex(Tuple rhs);

  /**
   * Create a {@link TupleIterator} that will contain at least the set of {@link Tuple}s that are needed to
   * completely join the lhs to the rhs based on the join.
   */
  public abstract TupleIterator createRhsIterator(Tuple lhs);

  /**
   * @return true if this join can be used, e.g. it's suitable for the query and join.  This is intended as a simpler
   * way of providing a set of implementations such that the {@link DefaultJoiner} can work through which
   * ones are appropriate by checking this method.
   */
  public abstract boolean isUsable();

}
