/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */

package nz.org.riskscape.engine.join;

import java.util.Arrays;
import java.util.List;

import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.rl.ExpressionRealizer;
import nz.org.riskscape.engine.rl.RealizedExpression;
import nz.org.riskscape.engine.types.Nullable;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.rl.ast.Expression;

import lombok.AccessLevel;
import lombok.Getter;
import lombok.RequiredArgsConstructor;

/**
 * {@link JoinCondition} based on a {@link RealizedExpression}
 */
@RequiredArgsConstructor(access = AccessLevel.PRIVATE)
public class ExpressionJoinCondition implements JoinCondition {

  public static ResultOrProblems<ExpressionJoinCondition> realize(Expression expression, Struct inputType,
      ExpressionRealizer eRealizer) {
    return eRealizer.realize(inputType, expression)
        .flatMap((re, p) -> {
          if (Nullable.strip(re.getResultType()) != Types.BOOLEAN) {
            return ResultOrProblems.failed(Problem.error("Expression '%s' must return Boolean. Found: %s",
                expression.toSource(), re.getResultType()));
          }
          return ResultOrProblems.of(new ExpressionJoinCondition(re, p), p);
        });
  }

  @Getter
  private final RealizedExpression expression;
  private final List<Problem> problems;

  @Override
  public boolean matches(Tuple row) {
    return expression.evaluate(row) == Boolean.TRUE;
  }

  @Override
  public List<Problem> validate(Struct joinedType) {
    if (joinedType != expression.getInputType()) {
      //This would be a programming error.
      return Arrays.asList(Problem.error("Joined type not the same instance as realized with"));
    }
    return problems;
  }

}
