/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.jai;

import java.util.List;

import javax.media.jai.JAI;

import org.apache.commons.math3.util.Precision;

import lombok.extern.slf4j.Slf4j;
import nz.org.riskscape.engine.Engine;
import nz.org.riskscape.engine.spi.EngineBootstrapper;

/**
 * Bootstrapper responsible for initializing the JAI tile cache with configured size and instrumentation.
 *
 * This bootstrapper configures the default JAI tile cache based on engine settings and wraps it
 * with an InstrumentedTileCache for performance monitoring.
 */
@Slf4j
public class TileCacheBootstrapper {

  public static final String TILE_CACHE_SETTING = "global.tile-cache-percent";
  public static final double DEFAULT_TILE_CACHE_PERCENT = 5D;

  /**
   * Creates an EngineBootstrapper for tile cache initialization.
   */
  public static final EngineBootstrapper INSTANCE =
      new EngineBootstrapper("tile-cache-initialization", TileCacheBootstrapper::initializeTileCache);

  /**
   * Initializes the tile cache with the configuration from the engine
   */
  private static void initializeTileCache(Engine.Writeable engine) {
    InstrumentedTileCache tileCache =
        new InstrumentedTileCache(JAI.getDefaultInstance().getTileCache(), engine.getDiagnostics().getMetricRegistry());

    long currCacheSize = tileCache.getMemoryCapacity();

    List<String> tileCacheSetting = engine.getSetting(TILE_CACHE_SETTING);

    double percent = tileCacheSetting.size() > 0 ? Double.parseDouble(tileCacheSetting.get(0))
        : DEFAULT_TILE_CACHE_PERCENT;

    long newCacheSize = (long) ((percent / 100) * Runtime.getRuntime().maxMemory());

    // only change it if it's an increase - this is a conservative thing, I think because the default is computed
    // heuristically?  Not entirely sure...
    if (newCacheSize > currCacheSize) {
      log.info("TileCache capacity set to {} MiB", Precision.round(newCacheSize / 1024.0 / 1024.0, 1));
      tileCache.setMemoryCapacity(newCacheSize);
    } else {
      log.info("TileCache capacity not changed, percentage was below JAI default");
    }

    // replace default with instrumented
    JAI.getDefaultInstance().setTileCache(tileCache);
  }
}
