/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.jai;

import java.awt.Point;
import java.awt.image.Raster;
import java.awt.image.RenderedImage;
import java.util.Comparator;

import javax.media.jai.TileCache;

import com.codahale.metrics.Meter;
import com.codahale.metrics.MetricRegistry;

import lombok.extern.slf4j.Slf4j;

/**
 * A decorator implementation of TileCache that logs cache hit/miss statistics for performance monitoring.
 *
 * @see TileCache
 * @see javax.media.jai.JAI
 */
@Slf4j
public class InstrumentedTileCache implements TileCache {

  private final TileCache delegate;

  private final Meter hitMeter;

  private final Meter missMeter;

  public InstrumentedTileCache(TileCache tileCache, MetricRegistry metricRegistry) {
    this.delegate = tileCache;

    this.hitMeter = metricRegistry.meter("tile-cache.hits");
    this.missMeter = metricRegistry.meter("tile-cache.misses");
  }

  @Override
  public Raster getTile(RenderedImage owner, int tileX, int tileY) {
    Raster tile = delegate.getTile(owner, tileX, tileY);
    if (tile != null) {
      hitMeter.mark();
      if (log.isDebugEnabled()) {
        log.debug("Tile HIT: (" + tileX + "," + tileY + ")");
      }
    } else {
      missMeter.mark();
      if (log.isDebugEnabled()) {
        log.debug("Tile MISS: (" + tileX + "," + tileY + ")");
      }
    }
    return tile;
  }

  // Delegate methods - forward all other TileCache operations to the underlying cache
  @Override
  public void add(RenderedImage owner, int tileX, int tileY, Raster data) {
    delegate.add(owner, tileX, tileY, data);
  }

  @Override
  public void add(RenderedImage owner, int tileX, int tileY, Raster data, Object tileCacheMetric) {
    delegate.add(owner, tileX, tileY, data, tileCacheMetric);
  }

  @Override
  public void addTiles(RenderedImage owner, Point[] tileIndices, Raster[] tiles, Object tileCacheMetric) {
    delegate.addTiles(owner, tileIndices, tiles, tileCacheMetric);
  }

  @Override
  public void remove(RenderedImage owner, int tileX, int tileY) {
    delegate.remove(owner, tileX, tileY);
  }

  @Override
  public void removeTiles(RenderedImage owner) {
    delegate.removeTiles(owner);
  }

  @Override
  public void flush() {
    delegate.flush();
  }

  @Override
  public long getMemoryCapacity() {
    return delegate.getMemoryCapacity();
  }

  @Override
  public void setMemoryCapacity(long memoryCapacity) {
    delegate.setMemoryCapacity(memoryCapacity);
  }

  @Override
  public int getTileCapacity() {
    return delegate.getTileCapacity();
  }

  @Override
  public void setTileCapacity(int tileCapacity) {
    delegate.setTileCapacity(tileCapacity);
  }

  @Override
  public float getMemoryThreshold() {
    return delegate.getMemoryThreshold();
  }

  @Override
  public void setMemoryThreshold(float threshold) {
    delegate.setMemoryThreshold(threshold);
  }

  @Override
  public Comparator getTileComparator() {
    return delegate.getTileComparator();
  }

  @Override
  public void setTileComparator(Comparator comparator) {
    delegate.setTileComparator(comparator);
  }

  @Override
  public Raster[] getTiles(RenderedImage owner) {
    return delegate.getTiles(owner);
  }

  @Override
  public Raster[] getTiles(RenderedImage owner, Point[] tileIndices) {
    return delegate.getTiles(owner, tileIndices);
  }

  @Override
  public void memoryControl() {
    delegate.memoryControl();
  }
}
