/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.io;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.file.Path;
import java.nio.file.Paths;

import java.util.regex.Pattern;

import com.codahale.metrics.Counter;

import lombok.Getter;

/**
 * {@link TupleStorage} backed by a directory on the filesystem.
 */
public class DiskStorage implements TupleStorage {

  public static final String TUPLES_FILE_FORMAT = "tuples-%05d";
  public static final Pattern TUPLES_FILE_PATTERN = Pattern.compile("tuples-[0-9]{3}");

  @Getter
  private final Path dataDir;
  private final boolean useCompression;
  private Counter counter;

  /**
   * Create disk storage that writes files in to the given directory
   * @param dataDir directory to store tuples in
   */
  public DiskStorage(Path dataDir) {
    this.dataDir = dataDir;
    this.useCompression = false;
  }

  /**
   * Create disk storage that writes out the to the JVM's temp dir.
   */
  public DiskStorage() {
    this.useCompression = false;
    this.dataDir = Paths.get(System.getProperty("java.io.tmpdir"));
  }

  /**
   * Create disk storage based on a given directory
   * @param dataDir directory to store tuples in
   * @param useCompression if true, all readers and writers from this storage will use gzip compression.   This can use
   * less disk at the cost of CPU when reading in and out.
   */
  public DiskStorage(Path dataDir, boolean useCompression, Counter counter) {
    this.dataDir = dataDir;
    this.useCompression = useCompression;
    this.counter = counter;
  }

  public DiskStorage(Counter counter) {
    this();
    this.counter = counter;
  }

  public DiskStorage(Path dataDir, boolean useCompression) {
    this(dataDir, useCompression, null);
  }

  @Override
  public TupleWriter createTemporaryWriter() throws IOException {

    /* Files.createTempFile REQUIRES a FileAttribute<?>. Setting it null fails with NPE.
       let's just do this.*/
    File ret = File.createTempFile("tuples", "", dataDir.toFile());
    ret.deleteOnExit();

    return new DiskStorageWriter(ret, useCompression, counter);
  }

  public DiskStorageReader getReader(DiskStorageWriter writer) throws IOException {
    if (!writer.getFile().exists()) {
      throw new FileNotFoundException("Attempting to create a reader from a writer with no file - "
          + "has it been cleaned up?");
    }
    return new DiskStorageReader(writer.getFile(), useCompression);
  }

  @Override
  public TupleReader getReader(TupleWriter writer) throws IOException {
    return getReader((DiskStorageWriter) writer);
  }

}
