/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.ini;

import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;

import com.google.common.collect.Maps;

import nz.org.riskscape.config.ConfigSection;
import nz.org.riskscape.engine.IdentifiedException;
import nz.org.riskscape.engine.Project;
import nz.org.riskscape.engine.bind.BindingContext;
import nz.org.riskscape.engine.bind.BoundParameters;
import nz.org.riskscape.engine.model.IdentifiedModel;
import nz.org.riskscape.engine.model.ModelFramework;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.Problems;
import nz.org.riskscape.problem.ResultOrProblems;

/**
 * Builds a framework-based model from an INI definition.
 */
public class IniFileModelBuilder2 extends IdentifiedObjectBuilder.Base<IdentifiedModel> {

  private final Project project;

  private static final String DEFAULT_DESCRIPTION = "";

  public static final String MODEL_KEY = "model";

  public IniFileModelBuilder2(Project project) {
    super(IdentifiedModel.class, MODEL_KEY);
    this.project = project;
  }

  /**
   * Builds a model from an INI file section
   */
  public ResultOrProblems<IdentifiedModel> buildModel(String modelName, ConfigSection section) {

    List<Problem> problems = new LinkedList<>();
    Consumer<List<Problem>> addProblems = ps -> problems.addAll(ps);

    // work out the model framework to use
    ResultOrProblems<String> frameworkId = section.getOneRequired("framework");
    if (!frameworkId.isPresent()) {
      return ResultOrProblems.failed(frameworkId.getProblems());
    }
    String description = section.getOne("description")
        .map(or -> or.orElse(addProblems, null))
        .orElse(DEFAULT_DESCRIPTION);

    ModelFramework framework;
    try {
      framework = project.getEngine().getModelFrameworks().get(frameworkId.get(), p -> problems.add(p));
    } catch (IdentifiedException e) {
      return ResultOrProblems.failed(e.getProblem());
    }

    if (Problem.hasErrors(problems)) {
      return ResultOrProblems.failed(problems);
    }

    // get all the INI section key values to bind, but lets remove the keys that we have used (framework/description)
    // so they won't end up looking like extraneous entries in the bound parameters.
    Map<String, List<String>> unbound = Maps.newHashMap(section.toMultiMap());
    unbound.remove("framework");
    unbound.remove("description");
    BindingContext bindingContext = project.newBindingContext().withNewRelativeTo(section.getLocation());

    // first we bind the parameters that the framework itself uses (i.e. things
    // like the template ID or pipeline definition, that can't be overridden later
    // once they've been specified)
    BoundParameters bound = framework.getBuildParameterSet().bind(bindingContext, ConfigSection.wild(unbound));

    // then use the resulting bound framework parameters to build the model.
    // If successful, wrap up the resulting model with its associated ID
    return bound
        .flatMap(buildParams -> framework.build(project, buildParams))
        .composeProblems((s, p) -> Problems.foundWith(IdentifiedModel.class, modelName, p))
        .map(model -> IdentifiedModel.wrap(modelName, description, model));
  }


  @Override
  public ResultOrProblems<IdentifiedModel> build(String id, ConfigSection section) {
    return buildModel(id, section);
  }

}
