/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.ini;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import nz.org.riskscape.config.ConfigProblems;
import nz.org.riskscape.config.ConfigSection;
import nz.org.riskscape.engine.IdentifiedException;
import nz.org.riskscape.engine.Project;
import nz.org.riskscape.engine.bind.BindingContext;
import nz.org.riskscape.engine.bind.BoundParameters;
import nz.org.riskscape.engine.bind.Parameter;
import nz.org.riskscape.engine.function.FunctionFramework;
import nz.org.riskscape.engine.function.IdentifiedFunction;
import nz.org.riskscape.engine.problem.GeneralProblems;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.ResultOrProblems;

/**
 * Builds a user-defined function from an INI definition.
 */
public class IniFileFunctionBuilder extends IdentifiedObjectBuilder.Base<IdentifiedFunction> {

  private final Project project;

  /**
   * The INI file keyword to denote a function section
   */
  public static final String FUNCTION_KEY = "function";

  public static final String FRAMEWORK_KEY = "framework";

  public IniFileFunctionBuilder(Project project) {
    super(IdentifiedFunction.class, FUNCTION_KEY);
    this.project = project;
  }

  private ResultOrProblems<IdentifiedFunction> buildFunction(String functionName, ConfigSection section) {
    // get all the INI section key values to bind
    List<Problem> warnings = new ArrayList<>();
    Map<String, String> sectionValues = section.toMap();

    // work out the framework (jython, classifier, etc) to use to build the function
    String frameworkId = sectionValues.getOrDefault(FRAMEWORK_KEY, getDefaultFrameworkId(section));
    if (frameworkId == null) {
      return ResultOrProblems.failed(GeneralProblems.required(FRAMEWORK_KEY, Parameter.class));
    }

    FunctionFramework framework = null;
    try {
      framework = project.getEngine().getFunctionFrameworks().get(frameworkId, p -> warnings.add(p));
    } catch (IdentifiedException ex) {
      return ResultOrProblems.error(ex);
    }

    // bind the parameters to check validity, arity, etc
    BindingContext bindingContext = project.newBindingContext().withNewRelativeTo(section.getLocation());
    BoundParameters bound = framework.getParameterSet()
        .bindConfig(bindingContext, section.toConfigMap());

    if (!bound.isValid()) {
      return ResultOrProblems.failed(bound.getValidationProblems());
    }

    // warn about any spurious keys in the INI section that we've ignored
    if (!bound.getExtraneous().isEmpty()) {
      warnings.add(ConfigProblems.get().spuriousKeys(section, bound.getExtraneous().keySet()));
    }

    return framework.build(functionName, project, bound).withMoreProblems(warnings);
  }

  private String getDefaultFrameworkId(ConfigSection section) {
    List<FunctionFramework> frameworks = new ArrayList<>(project.getEngine().getFunctionFrameworks().getAll());
    // sort descending
    frameworks.sort((a, b) -> Integer.compare(b.getPriority(), a.getPriority()));
    for (FunctionFramework framework : frameworks) {
      if (framework.canBuild(project, ConfigSection.wild(section.toMultiMap()))) {
        return framework.getId();
      }
    }
    return null;
  }

  @Override
  public ResultOrProblems<IdentifiedFunction> build(String id, ConfigSection section) {
    return buildFunction(id, section);
  }
}
