/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.ini;

import lombok.Getter;
import lombok.RequiredArgsConstructor;
import nz.org.riskscape.config.ConfigSection;
import nz.org.riskscape.engine.Identified;
import nz.org.riskscape.engine.IdentifiedCollection;
import nz.org.riskscape.engine.Project;
import nz.org.riskscape.engine.resource.Resource.PseudoResource;
import nz.org.riskscape.problem.ResultOrProblems;

/**
 * Builds an Identified RiskScape item (bookmark, type, model, etc) from an INI
 * file section.
 */
public interface IdentifiedObjectBuilder<T extends Identified> {

  @RequiredArgsConstructor
  abstract class Base<T extends Identified> implements IdentifiedObjectBuilder<T> {
    @Getter
    private final Class<T> identifiedClass;
    @Getter
    private final String keyword;
  }

  /**
   * @return the class of the Identified object being built
   */
  Class<T> getIdentifiedClass();

  /**
   * @return the keyword to look for in standalone INI file sections, e.g. "bookmark" in
   * ```
   * [bookmark foo]
   * bar = baz
   * ```
   */
  String getKeyword();

  /**
   * Builds the item based on the given INI section.
   *
   * @return the item successfully built, or any problems that occurred.
   */
  ResultOrProblems<T> build(String id, ConfigSection section);

  /**
   * Lazily adds the Identified item in the given INI section to the project. The
   * item will be built as needed, when accessed from the project collection.
   */
  default void addToProject(Project project, String id, ConfigSection section) {
    // add the identified item to the project so that it gets lazily built, as needed
    IdentifiedCollection<T> projectCollection = project.getCollectionByClass(getIdentifiedClass());
    projectCollection.add(id, new PseudoResource(section.getLocation()), () -> {
      return build(id, section);
    });
  }
}
