/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.i18n;

import java.text.MessageFormat;
import java.util.Arrays;
import java.util.Locale;
import java.util.ResourceBundle;
import java.util.ResourceBundle.Control;
import java.util.function.Function;
import java.util.regex.Pattern;

import com.google.common.base.Strings;

import lombok.RequiredArgsConstructor;
import nz.org.riskscape.problem.Problem;


/**
 * {@link MessageSource} implementation that will resolve messages:
 * - from many {@link ResourceBundle} baseNames
 * - will merge property files with the same base name across a classpath using {@link MergedPropertiesResourceBundle}
 * - will fallback through locales for values using built-in ResourceBundle parenting logic
 *
 * For a given message key bundle base names are searched in reverse. E.g the last base name added is searched first.
 */
@RequiredArgsConstructor
public class RiskscapeMessageSource implements MessageSource {

  /**
   * {@link Pattern} used to detect if a message contains and {@link MessageFormat} format specifiers.
   *
   * If no format specifiers are found then the message can be returned without performing any unnecessary
   * formatting.
   *
   * @deprecated detecting if the message is a format is only necessary for backwards compatibility
   * with {@link Problem}s which is a {@link MessageKey} but default messages for problems may contain
   * { or }. This handling should be removed.
   */
  @Deprecated
  public static final Pattern MESSAGE_FORMAT_DETECTOR = Pattern.compile("(\\{\\d(\\s*,\\s*\\w+\\s*){0,2}\\})+");

  /**
   * Gets a resource bundle using riskscape's customized {@link Control} rules.
   * @see ResourceBundleControl ResourceBundleControl
   */
  public static ResourceBundle getBundle(String baseName, Locale locale, ClassLoader classLoader) {
    return ResourceBundle.getBundle(baseName, locale, classLoader, ResourceBundleControl.INSTANCE);
  }

  private final Function<Locale, ResourceBundle> resourceBundleFunction;

  public RiskscapeMessageSource(String baseName, ClassLoader classLoader) {
    this(locale -> getBundle(baseName, locale, classLoader));
  }

  @Override
  public String getMessage(MessageKey resolvable) {
    return getMessage(resolvable, Locale.getDefault());
  }

  @Override
  public String getMessage(MessageKey resolvable, Locale locale) {
    if (resolvable.getCode() != null) {
      String result = getMessage(resolvable.getCode(), resolvable.getMessageArguments(), locale);
      if (result != null) {
        return result;
      }
    }
    return formatIfNecessary(resolvable.getDefaultMessage(), resolvable.getMessageArguments(), locale);
  }

  @Override
  public String getMessage(String code, Object... args) {
    return getMessage(code, args, Locale.getDefault());
  }

  @Override
  public String getMessage(String code, Object[] args, Locale locale) {
    return getMessage(code, args, null, locale);
  }

  @Override
  public String getMessage(String code, Object[] args, String defaultMessage, Locale locale) {
    //code may be null of synthetic {@link RiskscapeMessage}s, eg those that only have a default.
    ResourceBundle bundle = resourceBundleFunction.apply(locale);
    if (code != null) {
      if (bundle.containsKey(code)) {
        return formatIfNecessary(bundle.getString(code), args, locale);
      }
    }

    return formatIfNecessary(defaultMessage, args, locale);
  }

  String formatIfNecessary(String messageOrFormat, Object[] args, Locale locale) {
    if (Strings.isNullOrEmpty(messageOrFormat)
        || ! MESSAGE_FORMAT_DETECTOR.matcher(messageOrFormat).find()
        || args == null
        || args.length == 0) {
      return messageOrFormat;
    }
    try {
      return new MessageFormat(messageOrFormat).format(resolveArguments(args, locale));
    } catch (IllegalArgumentException e) {
      //messageOrFormat is not a format despite looking like it could be.
      //most likely this is from a Problem that contains {xxx} in the actual output message that gets
      //past the MESSAGE_FORMAT_DETECTOR. Hopefully this is unlikely
      return messageOrFormat;
    }
  }

  /**
   * Processes the args list to find any objects that are {@link MessageKey} and resolve them.
   * @param args to resolve
   * @return resolved args
   */
  private Object[] resolveArguments(Object[] args, Locale locale) {
    if (args == null || args.length == 0) {
      return args;
    }
    Object[] processed = Arrays.copyOf(args, args.length);
    for (int i = 0; i < args.length; i++) {
      Object arg = args[i];
      if (arg instanceof MessageKey) {
        processed[i] = getMessage((MessageKey)arg, locale);
      }
    }
    return processed;
  }

  public ResourceBundle getBundle(Locale l) {
    return resourceBundleFunction.apply(l);
  }

  /**
   * Returns an implementation of the given factory class that resolves messages against the given locale and message
   * source.  Defers to {@link MessageFactoryProxy}.
   */
  @Override
  public <T extends MessageFactory> T getMessageFactory(Class<T> factory, Locale locale) {
    return MessageFactoryProxy.getMessageFactory(factory, locale, this);
  }
}
