/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.i18n;

import java.io.IOException;
import java.net.URL;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.ResourceBundle;

import lombok.AccessLevel;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

/**
 * Extends Java's default {@link java.util.ResourceBundle.Control} to:
 * * Assumes properties files are encoded in utf-8 instead of latin-1
 * * Support loading the same named file from different locations to take the super-set of values from all, but with
 * last-one-wins behavior, where last-one is the last resource identified by the classloader as having that name
 * * Disables fall-back locale behavior so that this can be handled in the message source
 *
 */
@Slf4j
@RequiredArgsConstructor(access = AccessLevel.PRIVATE)
public class ResourceBundleControl extends ResourceBundle.Control {

  public static final ResourceBundleControl INSTANCE = new ResourceBundleControl();

  /**
   * Charset to load properties files in.  Avoids translators having to put escape codes in properties files.  Java9
   * does this by default as well.
   */
  public static final String PROPS_FILE_CHARSET = "UTF-8";

  @Override
  public List<String> getFormats(String baseName) {
    if (baseName == null) {
      throw new NullPointerException();
    }
    return FORMAT_PROPERTIES;
  }

  @Override
  public Locale getFallbackLocale(String baseName, Locale locale) {
    //Prevent fallback to the system default locale.
    return null;
  }

  @Override
  public ResourceBundle newBundle(String baseName,
      Locale locale,
      String format,
      ClassLoader loader,
      boolean reload) throws IllegalAccessException, InstantiationException, IOException {
    String bundleName = toBundleName(baseName, locale);
    String resourceName = toResourceName(bundleName, "properties");

    List<URL> urls;

    if (loader instanceof I18nClassLoader) {
      I18nClassLoader i18nLoader = (I18nClassLoader) loader;
      urls = i18nLoader.getResourcesByPrecedence(resourceName);
    } else if (loader instanceof ResourceClassLoader) {
      URL url = loader.getResource(resourceName);
      urls = url == null ? Collections.emptyList() : Collections.singletonList(url);
    } else {
      throw new IllegalArgumentException("Unsupported classloader - the riskscape ResourceBundleControl expects either"
          + " an I18nClassLoader or a ResourceClassLoader, but we were given a " + loader.getClass());
    }

    log.info("Created resource bundle for base+locale {}+{} from {}", baseName, locale, urls);

    return new MergedPropertiesResourceBundle(urls, PROPS_FILE_CHARSET);
  }



}
