/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.i18n;

import java.io.IOException;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.List;
import java.util.Properties;
import java.util.ResourceBundle;
import java.util.Set;

import lombok.extern.slf4j.Slf4j;

/**
 * {@link ResourceBundle} whose content is obtained from one or more property files that are effectively merged
 * together.
 *
 * The resolution priority is in the order of propsUrls, first found wins.
 *
 * {@link MergedPropertiesResourceBundle} is obtained by calling
 * {@link ResourceBundle#getBundle(java.lang.String, java.util.Locale, java.lang.ClassLoader,
 java.util.ResourceBundle.Control) with
 * {@link I18nClassLoader} class loader and {@link ResourceBundleControl}.
 */
@Slf4j
class MergedPropertiesResourceBundle extends ResourceBundle {

  private final List<URL> propsUrls;
  private final Properties cachedProperties = new Properties();
  private final List<Properties> searchThrough;

  /**
   * Creates a MergedPropertiesResourceBundle that merges the specified property file URLs.
   *
   * @param propsUrls list of URLs to property files
   * @param charsetName charset to read property files with
   * @throws UnsupportedEncodingException if charsetName is not supported
   * @throws IOException if an I/O exception occurs.
   */
  MergedPropertiesResourceBundle(List<URL> propsUrls, String charsetName)
      throws UnsupportedEncodingException, IOException {
    this.propsUrls = propsUrls;
    this.searchThrough = new ArrayList<>();
    for (URL url : propsUrls) {
      Properties props = new Properties();
      props.load(new InputStreamReader(url.openStream(), charsetName));
      searchThrough.add(props);
    }
  }

  @Override
  protected Object handleGetObject(String key) {
    return cachedProperties.computeIfAbsent(key, k -> searchFor(k));
  }

  private Object searchFor(Object key) {
    log.debug("Looking for key {}", key);
    for (Properties props : searchThrough) {
      Object found = props.get(key);

      if (found != null) {
        log.debug("Found key in {}", propsUrls.get(searchThrough.indexOf(props)));
        return found;
      }
    }
    return null;
  }

  @Override
  public Enumeration<String> getKeys() {
    return Collections.enumeration(getAllKeys());
  }

  protected Set<String> getAllKeys() {
    Set<String> combined = new HashSet<>();
    for (Properties properties : searchThrough) {
      for (Object key : properties.keySet()) {
        combined.add(key.toString());
      }
    }
    if (parent != null) {
      //Parent will always be a MergedPropertiesResourceBundle because ResourceBundleCONTROL
      //will only create them.
      combined.addAll(((MergedPropertiesResourceBundle)parent).getAllKeys());
    }

    return combined;
  }

}
