/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.i18n;

import java.util.ArrayList;
import java.util.List;

import com.google.common.base.CaseFormat;

import lombok.Data;
import lombok.RequiredArgsConstructor;

/**
 * Helper for handling translation of Enum values for user help, CLI display, etc.
 */
@RequiredArgsConstructor
public class EnumTranslator {

  private final TranslationContext context;

  /**
   * Handles accessing translated strings for a specific enum value
   */
  @Data
  public class EnumEntry<T extends Enum<?>> {

    /**
     * The enum value itself
     */
    private final T value;

    /**
     * Returns a translated 'label' string for this enum. If no i18n message was found, then the
     * {@link Enum#name()} is used (changed to lower-case hyphens, so it looks more parameter-esque).
     */
    public String getLabel(Object... args) {
      String label = context.getMessage(getCode(), args);
      if (label == null) {
        // no i18n message found. Use the enum name, but try to make it look more user-facing
        label = CaseFormat.UPPER_UNDERSCORE.to(CaseFormat.LOWER_HYPHEN, value.name());
      }
      return label;
    }

    /**
     * Same as {@link #getLabel(Object...)} except the `defaultMessage` will be used
     * (rather than the {@link Enum#name()} if no i18n message was found.
     */
    public String getLabelOr(String defaultMessage, Object... args) {
      String label = context.getMessage(getCode(), args);
      if (label == null) {
        label = defaultMessage;
      }
      return label;
    }

    /**
     * @return the base i18n code for this enum value
     */
    public String getCode() {
      return value.getClass().getCanonicalName() + "." + value.name();
    }

    /**
     * @return a more detailed description for the enum value, e.g. help associated with what it is or does.
     */
    public String getDescription(Object... args) {
      return context.getMessage(getCode() + ".description", args);
    }
  }

  /**
   * @return a list of {@link EnumEntry}s that correspond to the constants/values for the given enum.
   *         This makes it easier to access/display the translated text for an enum.
   */
  public <T extends Enum<?>> List<EnumEntry<T>> getEntries(Class<T> enumClass) {
    List<EnumEntry<T>> entries = new ArrayList<>();

    for (Object value : enumClass.getEnumConstants()) {
      entries.add(new EnumEntry<T>((T) value));
    }
    return entries;
  }

  /**
   * @return a single EnumEntry for the given enumValue, linked to this EnumTranslator's i18n state.
   */
  public <T extends Enum<?>> EnumEntry<T> getEntry(T enumValue) {
    return new EnumEntry<T>(enumValue);
  }
}
