/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.gt;

import java.util.List;
import java.util.stream.Collectors;

import org.geotools.factory.CommonFactoryFinder;
import org.geotools.filter.spatial.BBOXImpl;
import org.geotools.api.filter.And;
import org.geotools.api.filter.ExcludeFilter;
import org.geotools.api.filter.Filter;
import org.geotools.api.filter.FilterFactory;
import org.geotools.api.filter.FilterVisitor;
import org.geotools.api.filter.Id;
import org.geotools.api.filter.IncludeFilter;
import org.geotools.api.filter.Not;
import org.geotools.api.filter.Or;
import org.geotools.api.filter.PropertyIsBetween;
import org.geotools.api.filter.PropertyIsEqualTo;
import org.geotools.api.filter.PropertyIsGreaterThan;
import org.geotools.api.filter.PropertyIsGreaterThanOrEqualTo;
import org.geotools.api.filter.PropertyIsLessThan;
import org.geotools.api.filter.PropertyIsLessThanOrEqualTo;
import org.geotools.api.filter.PropertyIsLike;
import org.geotools.api.filter.PropertyIsNil;
import org.geotools.api.filter.PropertyIsNotEqualTo;
import org.geotools.api.filter.PropertyIsNull;
import org.geotools.api.filter.expression.Add;
import org.geotools.api.filter.expression.Divide;
import org.geotools.api.filter.expression.Expression;
import org.geotools.api.filter.expression.ExpressionVisitor;
import org.geotools.api.filter.expression.Function;
import org.geotools.api.filter.expression.Literal;
import org.geotools.api.filter.expression.Multiply;
import org.geotools.api.filter.expression.NilExpression;
import org.geotools.api.filter.expression.PropertyName;
import org.geotools.api.filter.expression.Subtract;
import org.geotools.api.filter.spatial.BBOX;
import org.geotools.api.filter.spatial.Beyond;
import org.geotools.api.filter.spatial.BinarySpatialOperator;
import org.geotools.api.filter.spatial.Contains;
import org.geotools.api.filter.spatial.Crosses;
import org.geotools.api.filter.spatial.DWithin;
import org.geotools.api.filter.spatial.Disjoint;
import org.geotools.api.filter.spatial.Equals;
import org.geotools.api.filter.spatial.Intersects;
import org.geotools.api.filter.spatial.Overlaps;
import org.geotools.api.filter.spatial.Touches;
import org.geotools.api.filter.spatial.Within;
import org.geotools.api.filter.temporal.After;
import org.geotools.api.filter.temporal.AnyInteracts;
import org.geotools.api.filter.temporal.Before;
import org.geotools.api.filter.temporal.Begins;
import org.geotools.api.filter.temporal.BegunBy;
import org.geotools.api.filter.temporal.During;
import org.geotools.api.filter.temporal.EndedBy;
import org.geotools.api.filter.temporal.Ends;
import org.geotools.api.filter.temporal.Meets;
import org.geotools.api.filter.temporal.MetBy;
import org.geotools.api.filter.temporal.OverlappedBy;
import org.geotools.api.filter.temporal.TContains;
import org.geotools.api.filter.temporal.TEquals;
import org.geotools.api.filter.temporal.TOverlaps;

import com.google.common.collect.Lists;

import org.geotools.api.filter.BinaryLogicOperator;

/**
 * Base class to build filter converters upon.
 *
 * When used directly the input {@link Filter} or {@link Expression} will in effect be cloned.
 */
public class ConvertFilter implements ExpressionVisitor, FilterVisitor {

  public final FilterFactory filterFactory;

  public ConvertFilter() {
    this.filterFactory = CommonFactoryFinder.getFilterFactory();
  }

  public ConvertFilter(FilterFactory filterFactory) {
    this.filterFactory = filterFactory;
  }

  @Override
  public Object visit(Multiply expression, Object extraData) {
    return filterFactory.multiply(
        (Expression) expression.getExpression1().accept(this, extraData),
        (Expression) expression.getExpression2().accept(this, extraData));
  }

  /**
   * Determine if visitSpatialOperator should be invoked in favour of interface-specific versions of this method.
   * This will be checked upon each {@link Filter} that implements {@link BinarySpatialOperator} being encountered,
   * so that implementations can switch this on and off as traversal proceeds.
   */
  protected boolean supportsVisitSpatialOperator() {
    return false;
  }

  /**
   * Visit a {@link Filter} that implements {@link BinarySpatialOperator}.  Only called if
   * {@link #supportsVisitSpatialOperator()} returns true.
   */
  protected Object visitSpatialOperator(BinarySpatialOperator operator, Object extraData) {
    return null;
  }

  @Override
  public Object visit(Literal expression, Object extraData) {
    return expression;
  }

  @Override
  public Object visit(Function expression, Object extraData) {
    List<Expression> newArgs = expression.getParameters().stream()
        .map(expr -> (Expression) expr.accept(this, extraData))
        .collect(Collectors.toList());

    return filterFactory.function(expression.getName(), newArgs.toArray(new Expression[0]));
  }

  @Override
  public Object visit(Divide expression, Object extraData) {
    return filterFactory.divide(
        (Expression) expression.getExpression1().accept(this, extraData),
        (Expression) expression.getExpression2().accept(this, extraData));
  }

  @Override
  public Object visit(Add expression, Object extraData) {
    return filterFactory.add(
        (Expression) expression.getExpression1().accept(this, extraData),
        (Expression) expression.getExpression2().accept(this, extraData));
  }

  @Override
  public Object visit(NilExpression expression, Object extraData) {
    return expression;
  }
  @Override
  public Object visit(Subtract expression, Object extraData) {
    return filterFactory.subtract(
        (Expression) expression.getExpression1().accept(this, extraData),
        (Expression) expression.getExpression2().accept(this, extraData));
  }

  @Override
  public Object visitNullFilter(Object extraData) {
    return null;
  }

  @Override
  public Object visit(ExcludeFilter filter, Object extraData) {
    return filter;
  }

  @Override
  public Object visit(IncludeFilter filter, Object extraData) {
    return filter;
  }

  @Override
  public Object visit(And filter, Object extraData) {
    return filterFactory.and(convertChildren(filter, extraData));
  }

  @Override
  public Object visit(Id filter, Object extraData) {
    throw new RuntimeException("Unsupported filter " + filter);
  }

  @Override
  public Object visit(Not filter, Object extraData) {
    return filterFactory.not((Filter) filter.getFilter().accept(this, extraData));
  }

  @Override
  public Object visit(Or filter, Object extraData) {
    return filterFactory.or(convertChildren(filter, extraData));
  }

  private List<Filter> convertChildren(BinaryLogicOperator filter, Object extraData) {
    List<Filter> converted = Lists.newArrayListWithCapacity(filter.getChildren().size());
    for (Filter child : filter.getChildren()) {
      converted.add((Filter) child.accept(this, extraData));
    }
    return converted;
  }

  @Override
  public Object visit(PropertyIsBetween filter, Object extraData) {
    return filterFactory.between(
        (Expression) filter.getExpression().accept(this, extraData),
        (Expression) filter.getLowerBoundary().accept(this, extraData),
        (Expression) filter.getUpperBoundary().accept(this, extraData),
        filter.getMatchAction()
    );
  }

  @Override
  public Object visit(PropertyIsEqualTo filter, Object extraData) {
    return filterFactory.equal(
        (Expression) filter.getExpression1().accept(this, extraData),
        (Expression) filter.getExpression2().accept(this, extraData),
        filter.isMatchingCase(),
        filter.getMatchAction());
  }

  @Override
  public Object visit(PropertyIsNotEqualTo filter, Object extraData) {
    return filterFactory.notEqual(
        (Expression) filter.getExpression1().accept(this, extraData),
        (Expression) filter.getExpression2().accept(this, extraData),
        filter.isMatchingCase(),
        filter.getMatchAction());
  }

  @Override
  public Object visit(PropertyIsGreaterThan filter, Object extraData) {
    return filterFactory.greater(
        (Expression) filter.getExpression1().accept(this, extraData),
        (Expression) filter.getExpression2().accept(this, extraData),
        filter.isMatchingCase(),
        filter.getMatchAction());
  }

  @Override
  public Object visit(PropertyIsGreaterThanOrEqualTo filter, Object extraData) {
    return filterFactory.greaterOrEqual(
        (Expression) filter.getExpression1().accept(this, extraData),
        (Expression) filter.getExpression2().accept(this, extraData),
        filter.isMatchingCase(),
        filter.getMatchAction());
  }

  @Override
  public Object visit(PropertyIsLessThan filter, Object extraData) {
    return filterFactory.less(
        (Expression) filter.getExpression1().accept(this, extraData),
        (Expression) filter.getExpression2().accept(this, extraData),
        filter.isMatchingCase(),
        filter.getMatchAction());
  }

  @Override
  public Object visit(PropertyIsLessThanOrEqualTo filter, Object extraData) {
    return filterFactory.lessOrEqual(
        (Expression) filter.getExpression1().accept(this, extraData),
        (Expression) filter.getExpression2().accept(this, extraData),
        filter.isMatchingCase(),
        filter.getMatchAction());
  }

  @Override
  public Object visit(PropertyIsLike filter, Object extraData) {
    return filterFactory.like(
        (Expression) filter.getExpression().accept(this, extraData),
        filter.getLiteral(),
        filter.getWildCard(),
        filter.getSingleChar(),
        filter.getEscape(),
        filter.isMatchingCase(),
        filter.getMatchAction());
  }

  @Override
  public Object visit(PropertyIsNull filter, Object extraData) {
    return filterFactory.isNull((Expression) filter.getExpression().accept(this, extraData));
  }

  @Override
  public Object visit(PropertyIsNil filter, Object extraData) {
    return filterFactory.isNil(
        (Expression) filter.getExpression().accept(this, extraData),
        filter.getNilReason());
  }

  @Override
  public Object visit(BBOX filter, Object extraData) {

    if (this.supportsVisitSpatialOperator()) {
      return this.visitSpatialOperator(filter, extraData);
    } else {
      // XXX not using factory here, as it has an awkward interface
      return new BBOXImpl(
          (Expression) filter.getExpression1().accept(this, extraData),
          (Expression) filter.getExpression2().accept(this, extraData),
          filter.getMatchAction());
    }
  }

  @Override
  public Object visit(Beyond filter, Object extraData) {
    if (this.supportsVisitSpatialOperator()) {
      return this.visitSpatialOperator(filter, extraData);
    } else {
      return filterFactory.beyond(
          (Expression) filter.getExpression1().accept(this, extraData),
          (Expression) filter.getExpression2().accept(this, extraData),
          filter.getDistance(),
          filter.getDistanceUnits(),
          filter.getMatchAction());
    }
  }

  @Override
  public Object visit(Contains filter, Object extraData) {
    if (this.supportsVisitSpatialOperator()) {
      return this.visitSpatialOperator(filter, extraData);
    } else {
      Expression expression1 = (Expression) filter.getExpression1().accept(this, extraData);
      Expression expression2 = (Expression) filter.getExpression2().accept(this, extraData);

      return filterFactory.contains(expression1, expression2, filter.getMatchAction());
    }
  }

  @Override
  public Object visit(Crosses filter, Object extraData) {
    if (this.supportsVisitSpatialOperator()) {
      return this.visitSpatialOperator(filter, extraData);
    } else {
      Expression expression1 = (Expression) filter.getExpression1().accept(this, extraData);
      Expression expression2 = (Expression) filter.getExpression2().accept(this, extraData);

      return filterFactory.crosses(expression1, expression2, filter.getMatchAction());
    }
  }

  @Override
  public Object visit(Disjoint filter, Object extraData) {
    if (this.supportsVisitSpatialOperator()) {
      return this.visitSpatialOperator(filter, extraData);
    } else {
      Expression expression1 = (Expression) filter.getExpression1().accept(this, extraData);
      Expression expression2 = (Expression) filter.getExpression2().accept(this, extraData);

      return filterFactory.disjoint(expression1, expression2, filter.getMatchAction());
    }
  }

  @Override
  public Object visit(DWithin filter, Object extraData) {
    if (this.supportsVisitSpatialOperator()) {
      return this.visitSpatialOperator(filter, extraData);
    } else {
      return filterFactory.dwithin(
          (Expression) filter.getExpression1().accept(this, extraData),
          (Expression) filter.getExpression2().accept(this, extraData),
          filter.getDistance(), filter.getDistanceUnits(), filter.getMatchAction());
    }
  }

  @Override
  public Object visit(Equals filter, Object extraData) {
    if (this.supportsVisitSpatialOperator()) {
      return this.visitSpatialOperator(filter, extraData);
    } else {
      return filterFactory.equal(
          (Expression) filter.getExpression1().accept(this, extraData),
          (Expression) filter.getExpression2().accept(this, extraData),
          filter.getMatchAction());
    }
  }

  @Override
  public Object visit(Intersects filter, Object extraData) {
    if (this.supportsVisitSpatialOperator()) {
      return this.visitSpatialOperator(filter, extraData);
    } else {
      Expression expression1 = (Expression) filter.getExpression1().accept(this, extraData);
      Expression expression2 = (Expression) filter.getExpression2().accept(this, extraData);

      return filterFactory.intersects(expression1, expression2, filter.getMatchAction());
    }
  }

  @Override
  public Object visit(Overlaps filter, Object extraData) {
    if (this.supportsVisitSpatialOperator()) {
      return this.visitSpatialOperator(filter, extraData);
    } else {
      Expression expression1 = (Expression) filter.getExpression1().accept(this, extraData);
      Expression expression2 = (Expression) filter.getExpression2().accept(this, extraData);

      return filterFactory.overlaps(expression1, expression2, filter.getMatchAction());
    }
  }

  @Override
  public Object visit(Touches filter, Object extraData) {
    if (this.supportsVisitSpatialOperator()) {
      return this.visitSpatialOperator(filter, extraData);
    } else {
      Expression expression1 = (Expression) filter.getExpression1().accept(this, extraData);
      Expression expression2 = (Expression) filter.getExpression2().accept(this, extraData);

      return filterFactory.touches(expression1, expression2, filter.getMatchAction());
    }
  }

  @Override
  public Object visit(Within filter, Object extraData) {
    if (this.supportsVisitSpatialOperator()) {
      return this.visitSpatialOperator(filter, extraData);
    } else {
      Expression expression1 = (Expression) filter.getExpression1().accept(this, extraData);
      Expression expression2 = (Expression) filter.getExpression2().accept(this, extraData);

      return filterFactory.within(expression1, expression2, filter.getMatchAction());
    }
  }

  @Override
  public Object visit(PropertyName expression, Object extraData) {
    return filterFactory.property(expression.getPropertyName());
  }

  @Override
  public Object visit(After after, Object extraData) {
    throw new RuntimeException("Not yet implemented");
  }

  @Override
  public Object visit(AnyInteracts anyInteracts, Object extraData) {
    throw new RuntimeException("Not yet implemented");
  }

  @Override
  public Object visit(Before before, Object extraData) {
    throw new RuntimeException("Not yet implemented");
  }

  @Override
  public Object visit(Begins begins, Object extraData) {
    throw new RuntimeException("Not yet implemented");
  }

  @Override
  public Object visit(BegunBy begunBy, Object extraData) {
    throw new RuntimeException("Not yet implemented");
  }

  @Override
  public Object visit(During during, Object extraData) {
    throw new RuntimeException("Not yet implemented");
  }

  @Override
  public Object visit(EndedBy endedBy, Object extraData) {
    throw new RuntimeException("Not yet implemented");
  }

  @Override
  public Object visit(Ends ends, Object extraData) {
    throw new RuntimeException("Not yet implemented");
  }

  @Override
  public Object visit(Meets meets, Object extraData) {
    throw new RuntimeException("Not yet implemented");
  }

  @Override
  public Object visit(MetBy metBy, Object extraData) {
    throw new RuntimeException("Not yet implemented");
  }

  @Override
  public Object visit(OverlappedBy overlappedBy, Object extraData) {
    throw new RuntimeException("Not yet implemented");
  }

  @Override
  public Object visit(TContains contains, Object extraData) {
    throw new RuntimeException("Not yet implemented");
  }

  @Override
  public Object visit(TEquals equals, Object extraData) {
    throw new RuntimeException("Not yet implemented");
  }

  @Override
  public Object visit(TOverlaps contains, Object extraData) {
    throw new RuntimeException("Not yet implemented");
  }

}
