/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.geo;

import java.util.EnumMap;

import org.locationtech.jts.geom.Coordinate;
import org.locationtech.jts.geom.Envelope;

import lombok.Getter;

/**
 * Represents the four relative locations of rectangles formed by bisecting another rectangle along the X and Y axis.
 *
 * Here mostly to simplify/clarify the quad logic in various geometry algorithms
 */
public enum Quadrant {

  TL(HalfPlane.TOP, HalfPlane.LEFT),
  TR(HalfPlane.TOP, HalfPlane.RIGHT),
  BR(HalfPlane.BOTTOM, HalfPlane.RIGHT),
  BL(HalfPlane.BOTTOM, HalfPlane.LEFT);

  @Getter
  private HalfPlane vertical;

  @Getter
  private HalfPlane horizontal;

  /**
   * Partition the given envelope in to four new ones defined by the corners and centre of the given one, such that the
   * union of all four is the given one, without any intersection.
   */
  public static EnumMap<Quadrant, Envelope> partition(Envelope envelope) {
    EnumMap<Quadrant, Envelope> map = new EnumMap<>(Quadrant.class);
    Coordinate centre = envelope.centre();
    for (Quadrant quadrant : Quadrant.values()) {
      Envelope newEnvelope = new Envelope();
      newEnvelope.expandToInclude(centre);
      newEnvelope.expandToInclude(quadrant.getCoordinate(envelope));
      map.put(quadrant, newEnvelope);
    }
    return map;
  }

  Quadrant(HalfPlane vert, HalfPlane horiz) {
    this.vertical = vert;
    this.horizontal = horiz;
  }

  /**
   * @returns the coordinate of the given envelope in this quadrant around the envelope's centre.
   */
  public Coordinate getCoordinate(Envelope envelope) {
    Coordinate coord = new Coordinate();

    if (horizontal == HalfPlane.LEFT) {
      coord.x = envelope.getMinX();
    } else {
      coord.x = envelope.getMaxX();
    }

    if (vertical == HalfPlane.BOTTOM) {
      coord.y = envelope.getMinY();
    } else {
      coord.y = envelope.getMaxY();
    }

    return coord;
  }
}
