/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.geo;

import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.TopologyException;
import org.locationtech.jts.noding.Noder;
import org.locationtech.jts.operation.overlay.OverlayOp;
import org.locationtech.jts.operation.overlayng.OverlayNGRobust;

import lombok.RequiredArgsConstructor;

/**
 * Performs geometry overlay operations on two geometries. These overlay operations are done here
 * (instead of call lhs.intersection(rhs)) to allow greater control on the {@link Noder}
 * implementation that is used. This can greatly reduce the chance of an overlay operation resulting
 * in the dreaded {@link TopologyException}.
 *
 * Note that the geometries returned from these operations are not guaranteed to be homogeneous. If
 * homogeneous geometries are required see {@link GeometryUtils#removeNonFamilyMembers(Geometry, GeometryFamily) }.
 */
@RequiredArgsConstructor
public abstract class OverlayOperations {

  /**
   * {@link OverlayOperations} based on {@link OverlayNGRobust}.
   */
  private static class RobustOverlayOperations extends OverlayOperations {

    @Override
    protected Geometry apply(Geometry lhs, Geometry rhs, int overlayOp) {
      // Robust should be sufficient to meet our needs but only time will tell.
      // If the dreaded TopologyException persists then we could experiment with using OverlayNG with
      // difference Noder implementations.

      return OverlayNGRobust.overlay(lhs, rhs, overlayOp);
    }
  }

  /**
   * @return A general purpose {@link OverlayOperations}.
   */
  public static OverlayOperations get() {
    return new RobustOverlayOperations();
  }

  /**
   * Applies a geometric overlay operation to the left and right hand side geometries.
   *
   * @param lhs left hand side geometry
   * @param rhs right hand side geometry
   * @param overlayOp operation to perform
   * @return result of applying the overlay operation to lhs/rhs
   */
  protected abstract Geometry apply(Geometry lhs, Geometry rhs, int overlayOp);

  /**
   * @return the geometric intersection of lhs/rhs
   */
  public Geometry intersection(Geometry lhs, Geometry rhs) {
    return apply(lhs, rhs, OverlayOp.INTERSECTION);
  }

  /**
   * @return the geometric difference of lhs/rhs
   */
  public Geometry difference(Geometry lhs, Geometry rhs) {
    return apply(lhs, rhs, OverlayOp.DIFFERENCE);
  }

}
