/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.function.maths;

import java.util.List;
import java.util.Optional;

import nz.org.riskscape.engine.function.ArgumentList;
import nz.org.riskscape.engine.function.BaseRealizableFunction;
import nz.org.riskscape.engine.function.RiskscapeFunction;
import nz.org.riskscape.engine.rl.RealizationContext;
import nz.org.riskscape.engine.types.Nullable;
import nz.org.riskscape.engine.types.RSList;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.engine.util.RandomUtils;
import nz.org.riskscape.problem.ProblemException;
import nz.org.riskscape.rl.ast.FunctionCall;

/**
 * Used to randomly choose an item from a list. Helpful in models where
 * you have a set of known values, but you also want some randomness
 * in which value gets used.
 */
public class RandomChoice extends BaseRealizableFunction {

  public RandomChoice() {
    super(ArgumentList.create(
      "items",   RSList.LIST_ANYTHING,
      "weights", Nullable.of(RSList.create(Types.FLOATING)),
      "seed", Nullable.INTEGER),
      Nullable.ANYTHING);
  }

  @Override
  public RiskscapeFunction build(RealizationContext context, FunctionCall functionCall, List<Type> givenTypes)
      throws ProblemException {

    RSList items = arguments.getRequiredAs(givenTypes, 0, RSList.class).getOrThrow();
    boolean hasWeights = arguments.getArgument(functionCall, "weights").isPresent();
    boolean hasSeed = arguments.getArgument(functionCall, "seed").isPresent();

    return RiskscapeFunction.create(this, givenTypes, items.getContainedType(), (args) -> {
      List<?> from = (List<?>) args.get(0);
      Optional<Long> seed = Optional.ofNullable(hasSeed ? (Long) args.get(2) : null);

      if (hasWeights) {
        // user has supplied weights for a weighted random choice
        @SuppressWarnings("unchecked")
        List<Double> weights = (List<Double>) args.get(1);
        return RandomUtils.pick(from, weights, seed);
      } else {
        return RandomUtils.pick(from, seed);
      }
    });
  }
}
