/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */

package nz.org.riskscape.engine.function.maths;

import java.util.Arrays;
import java.util.List;
import java.util.Optional;

import nz.org.riskscape.engine.ArgsProblems;
import nz.org.riskscape.engine.RiskscapeException;
import nz.org.riskscape.engine.function.BaseMathsFunction;
import nz.org.riskscape.engine.function.RiskscapeFunction;
import nz.org.riskscape.engine.rl.RealizableFunction;
import nz.org.riskscape.engine.rl.RealizationContext;
import nz.org.riskscape.engine.rl.agg.AggregationFunction;
import nz.org.riskscape.engine.types.RSList;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.rl.ast.FunctionCall;

import lombok.Getter;

/**
 * Base class for math functions that can be used as an {@link AggregationFunction} and also called
 * directly with both single list argument (as an aggregating function) or with two numeric arguments.
 */
public abstract class OverloadedAggregatingMathsFunction extends BaseMathsFunction implements RealizableFunction {

  @Getter
  private final Optional<AggregationFunction> aggregationFunction;
  private final RealizableFunction relizableAggFunction;

  public OverloadedAggregatingMathsFunction(AggregationFunction aggFunction) {
    this.aggregationFunction = Optional.of(aggFunction);

    RiskscapeFunction adapted = AggregationFunction.asFunction(aggFunction);
    if (adapted instanceof RealizableFunction) {
      relizableAggFunction = (RealizableFunction) adapted;
    } else {
      throw new RiskscapeException("Underlying aggregation function is not realizable");
    }
  }

  /**
   * @return list of functions for the alternative numeric arguments that this function supports
   */
  protected abstract List<RiskscapeFunction> getAlternatives();

  @Override
  public List<Type> getArgumentTypes() {
    return Arrays.asList(Types.ANYTHING, Types.ANYTHING);
  }

  @Override
  public Type getReturnType() {
    return Types.ANYTHING;
  }

  private boolean doArgumentsMatch(List<Type> givenTypes, List<Type> overloadedTypes, RealizationContext context) {
    for (int i = 0; i < givenTypes.size(); i++) {
      // check if the types are assignable. this means we don't need to worry about wrapping types here.
      // (we'll let the function resolver sort out the rest)
      if (! context.getProject().getTypeSet().isAssignable(givenTypes.get(i), overloadedTypes.get(i))) {
        return false;
      }
    }
    return true;
  }

  @Override
  public ResultOrProblems<RiskscapeFunction> realize(RealizationContext context, FunctionCall functionCall,
      List<Type> givenTypes) {
    // if we were given a list, use the aggregation function
    if (givenTypes.size() == 1 && givenTypes.get(0).find(RSList.class).isPresent()) {
      return relizableAggFunction.realize(context, functionCall, givenTypes);
    } else if (givenTypes.size() != 2) {
      return ResultOrProblems.failed(ArgsProblems.get().wrongNumber(2, givenTypes.size()));
    }

    // otherwise find the numeric alternative that matches the given args
    for (RiskscapeFunction alternative : getAlternatives()) {
      if (doArgumentsMatch(givenTypes, alternative.getArgumentTypes(), context)) {
        return ResultOrProblems.of(alternative);
      }
    }

    return ResultOrProblems.failed(ArgsProblems.get().realizableDidNotMatch(this, givenTypes));
  }

}
