/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.function.lang;

import java.util.ArrayList;
import java.util.List;

import nz.org.riskscape.engine.ArgsProblems;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.function.ArgumentList;
import nz.org.riskscape.engine.function.BaseRealizableFunction;
import nz.org.riskscape.engine.function.RiskscapeFunction;
import nz.org.riskscape.engine.rl.RealizationContext;
import nz.org.riskscape.engine.rl.RealizedExpression;
import nz.org.riskscape.engine.rl.ScopedLambdaExpression;
import nz.org.riskscape.engine.types.LambdaType;
import nz.org.riskscape.engine.types.Nullable;
import nz.org.riskscape.engine.types.RSList;
import nz.org.riskscape.engine.types.ScopedLambdaType;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Struct.StructMember;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.Problems;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.rl.ast.ExpressionProblems;
import nz.org.riskscape.rl.ast.FunctionCall;
import nz.org.riskscape.rl.ast.Lambda;

/**
 * Riskscape implementation of a zip mapping function.  Takes a two lists and a lambda function, returns a new list.
 */
public class ZipLists extends BaseRealizableFunction {

  public ZipLists() {
    super(
      ArgumentList.create(
          "lhs", RSList.create(Nullable.ANYTHING),
          "rhs", RSList.create(Nullable.ANYTHING),
          // in the future we might want to make the lambda arg optional and do some kind of auto merge
          // of the two items. But there are many questions about what the correct behaviour ought to be
          // in that case. So lets wait until there is a concrete use for that.
          "lambda", new LambdaType("lhs_element", "rhs_element")
      ),
      RSList.LIST_ANYTHING
    );
  }

  @Override
  public ResultOrProblems<RiskscapeFunction> realize(RealizationContext context, FunctionCall functionCall,
      List<Type> givenArgumentTypes) {

    // NB we could avoid a lot of this type checking code by asking the function realizer to do it for us - we are
    // advertising argument types that need to be respected, regardless of being realizable - so maybe we need a
    // flag on the RealizableFunction interface to say whether we want our argument types to be enforced or not.
    if (arguments.size() != givenArgumentTypes.size()) {
      return ResultOrProblems.failed(
          Problems.get(ArgsProblems.class).wrongNumber(arguments.size(), givenArgumentTypes.size()));
    }

    List<Problem> realizationProblems = new ArrayList<>();

    RSList list1Type = givenArgumentTypes.get(0).find(RSList.class).orElse(null);
    if (list1Type == null) {
      realizationProblems.add(ArgsProblems.mismatch(getArguments().get(0), givenArgumentTypes.get(0)));
    }

    RSList list2Type = givenArgumentTypes.get(1).find(RSList.class).orElse(null);
    if (list2Type == null) {
      realizationProblems.add(ArgsProblems.mismatch(getArguments().get(1), givenArgumentTypes.get(1)));
    }

    ScopedLambdaType lambdaType = givenArgumentTypes.get(2).find(ScopedLambdaType.class).orElse(null);
    if (lambdaType == null) {
      realizationProblems.add(ArgsProblems.mismatch(getArguments().get(2), givenArgumentTypes.get(2)));
    } else if (lambdaType.getArity() != 2) {
      realizationProblems.add(ExpressionProblems.get().lambdaArityError(
          functionCall.getArguments().get(2).getExpression(),
          lambdaType.getArity(),
          2
      ));
    }

    if (! realizationProblems.isEmpty()) {
      // args are not good. we fail
      return ResultOrProblems.failed(realizationProblems);
    }

    // NB the same lambda NBs in MapList#realize also apply here
    Lambda lambda = (Lambda) functionCall.getArguments().get(2).getExpression();

    String list1ElementArgName = lambda.getArguments().get(0).getValue();
    String list2ElementArgName = lambda.getArguments().get(1).getValue();
    Struct lambdaScopeType = lambdaType.buildCallingScopeType(list1Type.getMemberType(), list2Type.getMemberType());
    ResultOrProblems<RealizedExpression> realizedLambdaOr =
        context.getExpressionRealizer().realize(lambdaScopeType, lambda.getExpression());

    if (realizedLambdaOr.hasErrors()) {
      return ResultOrProblems.failed(ExpressionProblems.get().failedToRealize(
          lambda.getExpression(),
          lambdaScopeType
      ).withChildren(realizedLambdaOr.getProblems()));
    }
    RealizedExpression realized = realizedLambdaOr.get();
    RSList newListType = RSList.create(realized.getResultType());

    return ResultOrProblems.of(RiskscapeFunction.create(this, givenArgumentTypes, newListType, args -> {
        ScopedLambdaExpression givenLambda = (ScopedLambdaExpression) args.get(2);

        List<?> list1 = (List<?>) args.get(0);
        List<?> list2 = (List<?>) args.get(1);
        List<Object> newList = new ArrayList<>(list1.size());

        // If lists are not the same size we only zip to the shortest which is consistent with Python
        // but maybe this should be an Eval exception
        for (int i = 0; i < list1.size() && i < list2.size(); i++) {
          // build the scope each time anew in case it's closed over
          Tuple lambdaScope = givenLambda.buildCallingScope(lambdaScopeType);
          StructMember list1ElementDestMember = lambdaScopeType.getEntry(list1ElementArgName);
          StructMember list2ElementDestMember = lambdaScopeType.getEntry(list2ElementArgName);
          lambdaScope.set(list1ElementDestMember, list1.get(i));
          lambdaScope.set(list2ElementDestMember, list2.get(i));

          newList.add(realized.evaluate(lambdaScope));
        }

        return newList;
    }, realized));
  }
}
