/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.function.lang;


import java.util.List;

import nz.org.riskscape.engine.ArgsProblems;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.function.ArgumentList;
import nz.org.riskscape.engine.function.BaseRealizableFunction;
import nz.org.riskscape.engine.function.RiskscapeFunction;
import nz.org.riskscape.engine.rl.RealizationContext;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.problem.ProblemException;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.rl.ast.FunctionCall;

public class RemoveAttr extends BaseRealizableFunction {
  public RemoveAttr() {
    super(ArgumentList.create(
            "original", Types.ANYTHING,
            "remove", Types.ANYTHING
            // We should advertise Struct.EmptyStruct here, but if we do that then the coercing means we actually
            // just get given empty structs, which is not very helpful. This also means that the error message if you
            // specify the wrong type is a little unhelpful.
        ),
        Struct.EMPTY_STRUCT
    );
  }

  @Override
  public ResultOrProblems<RiskscapeFunction> realize(RealizationContext context, FunctionCall functionCall,
                                                     List<Type> givenTypes) {
    return ProblemException.catching(() -> {
      if (givenTypes.size() != arguments.size()) {
        throw new ProblemException(ArgsProblems.get().wrongNumber(arguments.size(), givenTypes.size()));
      }

      Struct original = arguments.getRequiredAs(givenTypes, 0, Struct.class).getOrThrow();

      // Try interpreting as text first
      ResultOrProblems<String> text = functionCall
          .getArguments()
          .get(1)
          .evaluateConstant(context, String.class, Types.TEXT);

      List<String> toRemove;
      if (text.isPresent()) {
        toRemove = List.of(text.get());
      } else {
        Struct struct = arguments.getRequiredAs(givenTypes, 1, Struct.class).getOrThrow();
        toRemove = struct.getMembers().stream().map(Struct.StructMember::getKey).toList();
      }

      Struct output = original.remove(toRemove).getOrThrow();

      List<Struct.StructMember> toInclude = original
          .getMembers()
          .stream()
          .filter(member -> !toRemove.contains(member.getKey()))
          .toList();

      return RiskscapeFunction.create(this, givenTypes, output, args -> {
        Tuple originalTuple = (Tuple) args.get(0);
        Tuple outputTuple = new Tuple(output);

        for (int i = 0; i < toInclude.size(); i++) {
          Struct.StructMember attr = toInclude.get(i);
          outputTuple.set(i, originalTuple.fetch(attr));
        }

        return outputTuple;
      });
    });

  }
}
