/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.function.lang;

import java.util.List;

import nz.org.riskscape.engine.ArgsProblems;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.function.ArgumentList;
import nz.org.riskscape.engine.function.BaseRealizableFunction;
import nz.org.riskscape.engine.function.RiskscapeFunction;
import nz.org.riskscape.engine.rl.RealizationContext;
import nz.org.riskscape.engine.rl.RealizedExpression;
import nz.org.riskscape.engine.types.Nullable;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.types.TypeProblems;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.engine.typexp.TypeBuildingException;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.rl.ast.ExpressionProblems;
import nz.org.riskscape.rl.ast.FunctionCall;

public class NullOf extends BaseRealizableFunction {

  public NullOf() {
    super(
      ArgumentList.create("typedef", Types.TEXT),
      Nullable.ANYTHING
    );
  }

  @Override
  public ResultOrProblems<RiskscapeFunction> realize(RealizationContext context, FunctionCall functionCall,
      List<Type> givenTypes
  ) {
    if (givenTypes.size() != arguments.size()) {
      return ResultOrProblems.failed(ArgsProblems.get().wrongNumber(arguments.size(), givenTypes.size()));
    }

    RealizedExpression typeExpression =
        context.getExpressionRealizer().
        realize(Struct.EMPTY_STRUCT, functionCall.getArguments().get(0).getExpression())
        .orElse(null);

    if (typeExpression == null) {
      return ResultOrProblems.failed(ExpressionProblems.get().constantRequired(
          functionCall.getArguments().get(0).getExpression()));
    }

    if (typeExpression.getResultType() != Types.TEXT) {
      return ResultOrProblems.failed(
          TypeProblems.get().mismatch(functionCall, Types.TEXT, typeExpression.getResultType())
      );
    }
    String typedef = (String) typeExpression.evaluate(Tuple.EMPTY_TUPLE);

    Type builtType;
    try {
      builtType = context.getProject().getTypeBuilder().build(typedef);
    } catch (TypeBuildingException ex) {
      return ResultOrProblems.error(ex);
    }

    return ResultOrProblems.of(RiskscapeFunction.create(this, givenTypes, Nullable.of(builtType), a -> null));
  }

}
