/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.function.lang;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;

import lombok.RequiredArgsConstructor;
import nz.org.riskscape.engine.ArgsProblems;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.function.ArgumentList;
import nz.org.riskscape.engine.function.BaseRealizableFunction;
import nz.org.riskscape.engine.function.RiskscapeFunction;
import nz.org.riskscape.engine.rl.RealizationContext;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Struct.StructBuilder;
import nz.org.riskscape.engine.types.Struct.StructMember;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.types.TypeProblems;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.rl.ast.FunctionCall;

public class MergeStruct extends BaseRealizableFunction {

  public MergeStruct() {
    super(
      ArgumentList.create(
          // ideally this would advertise a struct, but the coercing stuffs that up
        "original", Types.ANYTHING,
        "replace", Types.ANYTHING
      ),
      Struct.EMPTY_STRUCT
    );
  }

  @RequiredArgsConstructor
  private class IndexInfo {
    final boolean fromReplace;
    final int index;
  }

  @Override
  public ResultOrProblems<RiskscapeFunction> realize(RealizationContext context, FunctionCall functionCall,
      List<Type> givenTypes) {


    if (givenTypes.size() != 2) {
      return ResultOrProblems.failed(ArgsProblems.get().wrongNumber(2, givenTypes.size()));
    }

    List<Problem> problems = new ArrayList<>();
    Struct original = getStruct(problems, functionCall, givenTypes, 0);
    Struct replace = getStruct(problems, functionCall, givenTypes, 1);

    if (Problem.hasErrors(problems)) {
      return ResultOrProblems.failed(problems);
    }

    StructBuilder builder = new StructBuilder();
    IndexInfo[] indices = setupLoop(builder, original, replace);
    Struct builtType = context.normalizeStruct(builder.build());

    return ResultOrProblems.of(RiskscapeFunction.create(this, givenTypes, builtType, args -> {
        Tuple originalTuple = (Tuple) args.get(0);
        Tuple replaceTuple = (Tuple) args.get(1);
        Tuple mergedTuple = new Tuple(builtType);

        int index = 0;
        // NB if it turns out to be quicker, we could build an array (or list) here and use a bulk setAll method
        for (IndexInfo indexInfo : indices) {
          Object assign = (indexInfo.fromReplace ? replaceTuple : originalTuple).fetch(indexInfo.index);
          mergedTuple.set(index++, assign);
        }

        return mergedTuple;
    }));
  }

  private IndexInfo[] setupLoop(StructBuilder builder, Struct original, Struct replace) {
    // I should be able to do this with streams, but toMap doesn't support map supplier without also specifying
    // merge, at which point, it's more work than this
    Map<String, StructMember> replaceMembers = new LinkedHashMap<>();
    for (StructMember member : replace.getMembers()) {
      replaceMembers.put(member.getKey(), member);
    }

    List<IndexInfo> replacements = new ArrayList<>();

    for (StructMember originalMember : original.getMembers()) {
      StructMember replaceMember = replaceMembers.remove(originalMember.getKey());
      boolean fromReplace = replaceMember != null;
      StructMember member = fromReplace ? replaceMember : originalMember;

      replacements.add(new IndexInfo(fromReplace, member.getIndex()));

      builder.add(member.getKey(), member.getType());
    }

    for (Entry<String, StructMember> appended : replaceMembers.entrySet()) {
      StructMember append = appended.getValue();
      replacements.add(new IndexInfo(true, append.getIndex()));
      builder.add(append.getKey(), append.getType());
    }

    return replacements.toArray(new IndexInfo[replacements.size()]);
  }

  private Struct getStruct(List<Problem> problems, FunctionCall functionCall, List<Type> givenTypes, int i)  {

    Optional<Struct> structOr = givenTypes.get(i).find(Struct.class);

    if (!structOr.isPresent()) {
      problems.add(
          TypeProblems.get().mismatch(
              functionCall.getArguments().get(i).getExpression(),
              Struct.EMPTY_STRUCT,
              givenTypes.get(i)
          )
      );
    }

    return structOr.orElse(null);
  }
}
