/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.function.lang;

import java.util.ArrayList;
import java.util.List;

import nz.org.riskscape.engine.ArgsProblems;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.function.ArgumentList;
import nz.org.riskscape.engine.function.BaseRealizableFunction;
import nz.org.riskscape.engine.function.FunctionArgument;
import nz.org.riskscape.engine.function.RiskscapeFunction;
import nz.org.riskscape.engine.rl.RealizationContext;
import nz.org.riskscape.engine.rl.RealizedExpression;
import nz.org.riskscape.engine.rl.ScopedLambdaExpression;
import nz.org.riskscape.engine.types.LambdaType;
import nz.org.riskscape.engine.types.Nullable;
import nz.org.riskscape.engine.types.RSList;
import nz.org.riskscape.engine.types.ScopedLambdaType;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Struct.StructMember;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.types.TypeProblems;
import nz.org.riskscape.problem.Problems;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.rl.ast.ExpressionProblems;
import nz.org.riskscape.rl.ast.FunctionCall;
import nz.org.riskscape.rl.ast.Lambda;

/**
 * Riskscape implementation of a list mapping function.  Takes a list and a lambda function, returns a new list.  Or, if
 * the given arg is not a list, it applies the lambda expression to the arg directly.  This allows a mapping expression
 * to be applied to scalar values to make pipeline fragments easier to re-use, i.e. the same function applying
 * expression can work on a list of hazard values or a single hazard value
 */
public class MapList extends BaseRealizableFunction {

  public MapList() {
    super(
        ArgumentList.fromArray(
            new FunctionArgument("iterable", RSList.create(Nullable.ANYTHING)),
            new FunctionArgument("foreach", new LambdaType("list_element"))
        ),
        RSList.LIST_ANYTHING
    );
  }
  @Override
  public ResultOrProblems<RiskscapeFunction> realize(RealizationContext context, FunctionCall functionCall,
      List<Type> givenArgumentTypes) {

    // NB we could avoid a lot of this type checking code by asking the function realizer to do it for us - we are
    // advertising argument types that need to be respected, regardless of being realizable - so maybe we need a
    // flag on the RealizableFunction interface to say whether we want our argument types to be enforced or not.
    if (arguments.size() != givenArgumentTypes.size()) {
      return ResultOrProblems.failed(
          Problems.get(ArgsProblems.class).wrongNumber(arguments.size(), givenArgumentTypes.size()));
    }

    // map will either apply to all members of a list, or apply to the atom we've been given
    boolean nullableList = Nullable.is(givenArgumentTypes.get(0));
    Type memberType = givenArgumentTypes.get(0)
        .findAllowNull(RSList.class)
        .map(a -> a.getMemberType())
        .orElse(givenArgumentTypes.get(0));

    ScopedLambdaType lambdaType = givenArgumentTypes.get(1).find(ScopedLambdaType.class).orElse(null);
    if (lambdaType == null) {
      return ResultOrProblems.failed(Problems.get(TypeProblems.class)
          .mismatch(functionCall.getArguments().get(1), arguments.get(1).getType(), givenArgumentTypes.get(1)));
    }

    if (lambdaType.getArity() != 1) {
      return ResultOrProblems.failed(ExpressionProblems.get().lambdaArityError(
          functionCall.getArguments().get(1).getExpression(),
          lambdaType.getArity(),
          1
      ));
    }

    // NB at the moment, only constant-lambdas are supported - dynamic ones would require a way of realizing types
    // at runtime, which is going to get hairy and seems like an edge case
    Lambda lambda = (Lambda) functionCall.getArguments().get(1).getExpression();

    // NB - as an optimization, we could probably check the expression to see if it references any properties
    // outside the argument list - if not, we can avoid building a scope object each time, which will cut down on a few
    // cycles.  Not sure if worth it?
    Struct lambdaScopeType = lambdaType.buildCallingScopeType(memberType);
    ResultOrProblems<RealizedExpression> realizedLambdaOr =
        context.getExpressionRealizer().realize(lambdaScopeType, lambda.getExpression());

    if (realizedLambdaOr.hasErrors()) {
      return ResultOrProblems.failed(ExpressionProblems.get().failedToRealize(
          lambda.getExpression(),
          lambdaScopeType
      ).withChildren(realizedLambdaOr.getProblems()));
    }
    RealizedExpression realized = realizedLambdaOr.get();

    // return a different function, depending on whether it's a list or not
    if (memberType == givenArgumentTypes.get(0)) {
      // we got a single item - just apply the lambda to the item directly
      return ResultOrProblems.of(RiskscapeFunction.create(this, givenArgumentTypes, realized.getResultType(), args -> {
        ScopedLambdaExpression givenLambda = (ScopedLambdaExpression) args.get(1);
        Tuple lambdaScope = givenLambda.buildCallingScope(lambdaScopeType, args.get(0));
        return realized.evaluate(lambdaScope);
      }));
    } else {
      // we got a list - apply the lambda to each item in the list
      Type newListType = Nullable.ifTrue(nullableList, RSList.create(realized.getResultType()));
      return ResultOrProblems.of(RiskscapeFunction.create(this, givenArgumentTypes, newListType, args -> {
          List<?> list = (List<?>) args.get(0);
          if (list == null) {
            return null;
          }

          ScopedLambdaExpression givenLambda = (ScopedLambdaExpression) args.get(1);

          // now either replace, or append, the list value
          List<Object> newList = new ArrayList<>(list.size());
          for (Object object : list) {
            // It's important here that we create a new scope for each call for the list - it's possible the expression
            // will use this tuple, in which case it'll be broken
            Tuple lambdaScope = givenLambda.buildCallingScope(lambdaScopeType);
            StructMember listElementDestMember = lambdaScopeType.getEntry(lambdaType.getArgs().get(0));
            lambdaScope.set(listElementDestMember, object);

            newList.add(realized.evaluate(lambdaScope));
          }

          return newList;
      }, realized));
    }

  }
}
