/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.function.lang;

import java.util.List;

import nz.org.riskscape.engine.ArgsProblems;
import nz.org.riskscape.engine.function.ArgumentList;
import nz.org.riskscape.engine.function.BaseRealizableFunction;
import nz.org.riskscape.engine.function.RiskscapeFunction;
import nz.org.riskscape.engine.rl.RealizationContext;
import nz.org.riskscape.engine.types.Nullable;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.types.TypeProblems;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.rl.ast.FunctionCall;

public class IfNull extends BaseRealizableFunction {

  public IfNull() {
    super(
      ArgumentList.create("test", Nullable.ANYTHING, "else", Nullable.ANYTHING),
      Types.ANYTHING
    );
  }

  @Override
  public ResultOrProblems<RiskscapeFunction> realize(RealizationContext context, FunctionCall functionCall,
      List<Type> givenTypes) {

    if (givenTypes.size() != arguments.size()) {
      return ResultOrProblems.failed(ArgsProblems.get().wrongNumber(arguments.size(), givenTypes.size()));
    }

    // NB warn on non null?
    Type testType = Nullable.strip(givenTypes.get(0));
    Type elseType = givenTypes.get(1);

    // we advertise the 'else' argument as Nullable to avoid the NullSafeFunction wrapper kicking in,
    // but in reality a nullable 'else' argument is just going to lead to pointless user confusion
    // when they call if_null(), but still end up with a null result
    if (elseType.isNullable()) {
      return ResultOrProblems.failed(TypeProblems.get().cannotBeNull(arguments.get(1), testType, elseType));
    }

    Type realizedReturnType;

    // Here we test the covariance of our two operands in both directions and then declare our return type for whatever
    // works.  This was specifically added to allow `if_null(some_list_type, [])`, but could be useful in other
    // situations where the else case is 'wider' than the tested type - not sure we have any of those at the moment, but
    // types are pluggable, so maybe?
    if (testType == Types.NOTHING // nothing is alway null so just use the test case
        || context.getProject().getTypeSet().isAssignable(testType, elseType)) {
      realizedReturnType = elseType;
    } else if (context.getProject().getTypeSet().isAssignable(elseType, testType)) {
      realizedReturnType = testType;
    } else {
      return ResultOrProblems.failed(
          TypeProblems.get().mismatch(functionCall.getArguments().get(0), elseType, testType));
    }

    return ResultOrProblems.of(RiskscapeFunction.create(this, givenTypes, realizedReturnType, args -> {
        Object test = args.get(0);
        if (test == null) {
          return args.get(1);
        } else {
          return test;
        }
    }));
  }


}
