/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.function.lang;

import java.util.ArrayList;
import java.util.List;

import nz.org.riskscape.engine.ArgsProblems;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.function.ArgumentList;
import nz.org.riskscape.engine.function.BaseRealizableFunction;
import nz.org.riskscape.engine.function.FunctionArgument;
import nz.org.riskscape.engine.function.RiskscapeFunction;
import nz.org.riskscape.engine.problem.ProblemFactory;
import nz.org.riskscape.engine.rl.RealizationContext;
import nz.org.riskscape.engine.rl.RealizedExpression;
import nz.org.riskscape.engine.rl.ScopedLambdaExpression;
import nz.org.riskscape.engine.types.LambdaType;
import nz.org.riskscape.engine.types.Nullable;
import nz.org.riskscape.engine.types.RSList;
import nz.org.riskscape.engine.types.ScopedLambdaType;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Struct.StructMember;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.ProblemException;
import nz.org.riskscape.problem.Problems;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.rl.ast.ExpressionProblems;
import nz.org.riskscape.rl.ast.FunctionCall;
import nz.org.riskscape.rl.ast.Lambda;

/**
 * Riskscape implementation of a list filtering function.
 */
public class FilterList extends BaseRealizableFunction {

  public interface LocalProblems extends ProblemFactory {

    Problem testLambdaRequireBooleanResult(Type actual);
  }

  public static final LocalProblems LOCAL_PROBLEMS = Problems.get(LocalProblems.class);

  public FilterList() {
    super(
        ArgumentList.fromArray(
            new FunctionArgument("list", RSList.create(Nullable.ANYTHING)),
            new FunctionArgument("test", new LambdaType("list_element"))
        ),
        RSList.LIST_ANYTHING
    );
  }

  @Override
  public ResultOrProblems<RiskscapeFunction> realize(RealizationContext context, FunctionCall functionCall,
      List<Type> givenArgumentTypes) {
    return ProblemException.catching(() -> {

      if (arguments.size() != givenArgumentTypes.size()) {
        throw new ProblemException(
            ArgsProblems.get().wrongNumber(arguments.size(), givenArgumentTypes.size()));
      }

      Type memberType = givenArgumentTypes.get(0)
          .findAllowNull(RSList.class)
          .map(a -> a.getMemberType())
          .orElseThrow(() -> new ProblemException(ArgsProblems.mismatch(arguments.get(0), givenArgumentTypes.get(0))));

      ScopedLambdaType lambdaType = givenArgumentTypes.get(1)
          .find(ScopedLambdaType.class)
          .orElseThrow(() -> new ProblemException(ArgsProblems.mismatch(arguments.get(1), givenArgumentTypes.get(1))));

      if (lambdaType.getArity() != 1) {
        throw new ProblemException(ExpressionProblems.get().lambdaArityError(
            functionCall.getArguments().get(1).getExpression(),
            lambdaType.getArity(),
            1
        ));
      }

      Lambda lambda = (Lambda) functionCall.getArguments().get(1).getExpression();

      Struct lambdaScopeType = lambdaType.buildCallingScopeType(memberType);
      RealizedExpression realized = context.getExpressionRealizer().realize(lambdaScopeType, lambda.getExpression())
          .getOrThrow((problems)
              -> ExpressionProblems.get().failedToRealize(lambda.getExpression(), lambdaScopeType)
              .withChildren(problems)
          );

      if (Nullable.strip(realized.getResultType()) != Types.BOOLEAN) {
        throw new ProblemException(LOCAL_PROBLEMS.testLambdaRequireBooleanResult(realized.getResultType()));
      }

      return RiskscapeFunction.create(this, givenArgumentTypes, givenArgumentTypes.get(0), args -> {
        List<?> list = (List<?>) args.get(0);
        if (list == null) {
          return null;
        }

        ScopedLambdaExpression givenLambda = (ScopedLambdaExpression) args.get(1);
        // create a placeholder scope - we set the list value during the loop
        Tuple lambdaScope = givenLambda.buildCallingScope(lambdaScopeType);
        StructMember listElementDestMember = lambdaScopeType.getEntry(lambdaType.getArgs().get(0));

        List<Object> newList = new ArrayList<>(list.size());
        for (Object object : list) {
          // update list element in scope
          lambdaScope.set(listElementDestMember, object);
          Boolean test = (Boolean) realized.evaluate(lambdaScope);
          if (test != null && test) {
            newList.add(object);
          }
        }

        return newList;
      });
    });
  }

}
