/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.function.lang;

import java.util.ArrayList;
import java.util.List;

import nz.org.riskscape.engine.ArgsProblems;
import nz.org.riskscape.engine.function.ArgumentList;
import nz.org.riskscape.engine.function.BaseRealizableFunction;
import nz.org.riskscape.engine.function.RiskscapeFunction;
import nz.org.riskscape.engine.rl.RealizationContext;
import nz.org.riskscape.engine.types.EmptyList;
import nz.org.riskscape.engine.types.Nullable;
import nz.org.riskscape.engine.types.RSList;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.engine.types.ancestor.AncestorType;
import nz.org.riskscape.problem.ProblemException;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.rl.ast.FunctionCall;
import nz.org.riskscape.util.ListUtils;

public class ConcatList extends BaseRealizableFunction {

  public ConcatList() {
    super(
      ArgumentList.create(
          "lhs", Nullable.of(RSList.LIST_ANYTHING),
          "rhs", Nullable.of(RSList.LIST_ANYTHING)
      ),
      RSList.LIST_ANYTHING
    );
  }

  @Override
  public ResultOrProblems<RiskscapeFunction> realize(RealizationContext context, FunctionCall functionCall,
      List<Type> givenTypes) {

    if (givenTypes.size() != arguments.size()) {
      return ResultOrProblems.failed(ArgsProblems.get().wrongNumber(arguments.size(), givenTypes.size()));
    }

    return ProblemException.catching(() -> {
      Type lhsType = Nullable.strip(givenTypes.get(0));
      Type rhsType = Nullable.strip(givenTypes.get(1));

      if (lhsType == EmptyList.INSTANCE) {
        return emptyFunction(functionCall, givenTypes, 1);
      }
      if (rhsType == EmptyList.INSTANCE) {
        return emptyFunction(functionCall, givenTypes, 0);
      }

      RSList lhsListType = Types.findOrThrow(functionCall.getArguments().get(0), RSList.LIST_ANYTHING, lhsType);
      RSList rhsListType = Types.findOrThrow(functionCall.getArguments().get(1), RSList.LIST_ANYTHING, rhsType);

      // work out the common type of the 2 lists
      AncestorType ancestorType = context.getTypeSet().computeAncestorType(
          lhsListType,
          rhsListType
      ).orElse(AncestorType.of(RSList.create(
          // if either list types contain nullable items then we'll need a nullable anything
          Nullable.ifTrue(
              lhsListType.getContainedType().isNullable() || rhsListType.getContainedType().isNullable(),
              Types.ANYTHING
          )
      )));

      return RiskscapeFunction.create(this, givenTypes,
          ancestorType.getType(),
          args -> {
            List<?> lhs = (List<?>) ancestorType.getConvert().apply(args.get(0));
            List<?> rhs = (List<?>) ancestorType.getConvert().apply(args.get(1));

            if (lhs == null && rhs == null) {
              return new ArrayList<>();
            } else if (rhs == null || lhs == null) {
              return new ArrayList<>(lhs == null ? rhs : lhs);
            }

            // TODO probably want to consider whether some sort of immutable list type might be better here
            // to allow lists to be linked, which will work better with the immutable style of rl
            return ListUtils.concat(lhs, rhs);
          });
    });
  }

  private RiskscapeFunction emptyFunction(
      FunctionCall functionCall,
      List<Type> givenTypes,
      int constantIndex
  ) throws ProblemException {

    RSList listType = Types.findOrThrow(
        functionCall.getArguments().get(constantIndex),
        RSList.LIST_ANYTHING,
        givenTypes.get(constantIndex)
    );

    return new RiskscapeFunction() {

      @Override
      public Type getReturnType() {
        return listType;
      }

      @Override
      public List<Type> getArgumentTypes() {
        return givenTypes;
      }

      @Override
      public Object call(List<Object> args) {
        return args.get(constantIndex);
      }
    };
  }


}
