/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.function.lang;

import java.util.List;

import nz.org.riskscape.engine.ArgsProblems;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.function.ArgumentList;
import nz.org.riskscape.engine.function.BaseRealizableFunction;
import nz.org.riskscape.engine.function.RiskscapeFunction;
import nz.org.riskscape.engine.rl.RealizationContext;
import nz.org.riskscape.engine.rl.RealizedExpression;
import nz.org.riskscape.engine.types.LambdaType;
import nz.org.riskscape.engine.types.ScopedLambdaType;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.types.TypeProblems;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.problem.Problems;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.rl.ast.ExpressionProblems;
import nz.org.riskscape.rl.ast.FunctionCall;
import nz.org.riskscape.rl.ast.Lambda;

/**
 * A RiskScape 'test' function that evaluates a simple (argument-less) lamba
 * expression with given scope/input.
 */
public class Call extends BaseRealizableFunction {

  public Call() {
    super(
      ArgumentList.create(
          "scope", Struct.EMPTY_STRUCT,
          "lambda", LambdaType.NO_ARGS
      ),
      Types.ANYTHING
    );
  }

  @Override
  public ResultOrProblems<RiskscapeFunction> realize(RealizationContext context, FunctionCall functionCall,
      List<Type> givenTypes) {

    if (arguments.size() != givenTypes.size()) {
      return ResultOrProblems.failed(
          Problems.get(ArgsProblems.class).wrongNumber(arguments.size(), givenTypes.size()));
    }

    Struct inputType = givenTypes.get(0).asStruct();
    ScopedLambdaType lambdaType = givenTypes.get(1).find(ScopedLambdaType.class).orElse(null);
    if (lambdaType == null) {
      return ResultOrProblems.failed(Problems.get(TypeProblems.class)
          .mismatch(functionCall.getArguments().get(1), arguments.get(1).getType(), givenTypes.get(1)));
    }

    if (lambdaType.getArity() != 0) {
      return ResultOrProblems.failed(ExpressionProblems.get().lambdaArityError(
          functionCall.getArguments().get(1).getExpression(),
          lambdaType.getArity(),
          0
      ));
    }

    Lambda lambda = (Lambda) functionCall.getArguments().get(1).getExpression();
    ResultOrProblems<RealizedExpression> realizedOr = context.getExpressionRealizer().realize(inputType,
        lambda.getExpression());

    if (realizedOr.hasErrors()) {
      return ResultOrProblems.failed(
          ExpressionProblems.get().failedToRealize(lambda.getExpression(), inputType)
            .withChildren(realizedOr.getProblems())
      );
    }
    RealizedExpression realizedLambda = realizedOr.get();

    return ResultOrProblems.of(RiskscapeFunction.create(this, givenTypes, realizedLambda.getResultType(), args -> {
        Object arg = args.get(0);

        if (arg instanceof Tuple) {
          return realizedLambda.evaluate(arg);
        } else {
          return realizedLambda.evaluate(Tuple.ofValues(inputType, arg));
        }
    }));
  }
}
