/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.function.lang;

import java.util.List;
import java.util.Optional;

import nz.org.riskscape.engine.ArgsProblems;
import nz.org.riskscape.engine.RiskscapeException;
import nz.org.riskscape.engine.bind.ParameterField;
import nz.org.riskscape.engine.function.ArgumentList;
import nz.org.riskscape.engine.function.BaseRealizableFunction;
import nz.org.riskscape.engine.function.FunctionArgument;
import nz.org.riskscape.engine.function.RiskscapeFunction;
import nz.org.riskscape.engine.problem.ProblemFactory;
import nz.org.riskscape.engine.rl.RealizationContext;
import nz.org.riskscape.engine.types.Nullable;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.engine.util.FunctionCallOptions;
import nz.org.riskscape.engine.util.Pair;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.ProblemException;
import nz.org.riskscape.problem.Problems;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.rl.ast.Expression;
import nz.org.riskscape.rl.ast.FunctionCall;

public class AssertNotNull extends BaseRealizableFunction {

  public interface LocalProblems extends ProblemFactory {

    Problem unexpectedNull(int lineno, Expression itemExpression);

    Problem unexpectedNullCustomMessage(String message);
  }

  public static final LocalProblems PROBLEMS = Problems.get(LocalProblems.class);

  public static class Options {

    @ParameterField
    public Optional<String> message = Optional.empty();
  }

  public AssertNotNull() {
    super(
        ArgumentList.fromArray(
            new FunctionArgument("value", Nullable.ANYTHING),
            FunctionCallOptions.options(Options.class)
        ),
        Types.ANYTHING
    );
  }

  @Override
  public ResultOrProblems<RiskscapeFunction> realize(RealizationContext context, FunctionCall functionCall,
      List<Type> givenTypes) {
    return ProblemException.catching(() -> {
      if (givenTypes.size() < 1 || givenTypes.size() > arguments.size()) {
        throw new ProblemException(ArgsProblems.get().wrongNumberRange(1, arguments.size(), givenTypes.size()));
      }

      // strip nulls off the given type. this will be the return type.
      Type itemType = Nullable.strip(givenTypes.get(0));

      Options options = FunctionCallOptions.bindOptions(Options.class, context, arguments, functionCall).getOrThrow();

      return RiskscapeFunction.create(this, givenTypes, itemType, args -> {
        Object item = args.get(0);
        if (item == null) {
          String customMessage = options.message.orElse("");
          if (customMessage.isBlank()) {
            Expression itemExpression = functionCall.getArguments().get(0).getExpression();
            int line = itemExpression.getBoundary()
                .map(Pair::getLeft)
                .map(t -> t.getLocation().getLine())
                .orElse(1);
            throw new RiskscapeException(PROBLEMS.unexpectedNull(line, itemExpression));
          }
          throw new RiskscapeException(PROBLEMS.unexpectedNullCustomMessage(customMessage));
        } else {
          return item;
        }
      });
    });
  }

}
