/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package nz.org.riskscape.engine.function;

import java.net.URL;
import java.net.URLClassLoader;
import java.util.Arrays;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.List;

import nz.org.riskscape.engine.Engine;

/**
 * Each {@link JavaFunction} may have classes loaded from its own {@link FunctionClassLoader}.
 *
 * These should inherit from the {@link Engine}'s class loader, so that the functions can access the core classes.
 *
 * TODO User-defined functions are in this form a sort of Riskscape-sanctioned malware vector.  There
 * is a ticket, https://bugs.riskscape.org.nz/issues/434, that has a bunch of mitigations for making this
 * less ropey.
 */
public class FunctionClassLoader extends URLClassLoader {

  private URL[] urls;
  private ClassLoader parent;
  private ClassLoader system = getSystemClassLoader();

  /**
   * Construtor
   *
   * @param urls
   *          Array of urls with own libraries and all exported libraries of
   *          plugins that are required to this plugin
   * @param parent
   */
  public FunctionClassLoader(URL[] urls, ClassLoader parent) {
    super(urls, parent);

    this.urls = urls;
    this.parent = parent;
  }

  @Override
  protected synchronized Class<?> loadClass(String name, boolean resolve)
      throws ClassNotFoundException {

    // First, check if the class has already been loaded
    Class<?> c = findLoadedClass(name);

    if (c == null) {
      try {
        // checking local
        c = findClass(name);
      } catch (ClassNotFoundException | SecurityException e) {
        c = loadClassFromParent(name, resolve);
      }
    }

    if (resolve) {
      resolveClass(c);
    }

    return c;
  }

  private Class<?> loadClassFromParent(String name, boolean resolve)
      throws ClassNotFoundException {
    // checking parent
    // This call to loadClass may eventually call findClass
    // again, in case the parent doesn't find anything.
    Class<?> c;
    try {
      c = super.loadClass(name, resolve);
    } catch (ClassNotFoundException e) {
      c = loadClassFromSystem(name);
    } catch (SecurityException e) {
      c = loadClassFromSystem(name);
    }
    return c;
  }

  private Class<?> loadClassFromSystem(String name)
      throws ClassNotFoundException {
    Class<?> c = null;
    if (system != null) {
      // checking system: jvm classes, endorsed, cmd classpath,
      c = system.loadClass(name);
    }
    return c;
  }

  @Override
  public URL getResource(String name) {
    URL url = findResource(name);
    if (url == null) {
      url = super.getResource(name);
    }

    if (url == null && system != null) {
      url = system.getResource(name);
    }

    return url;
  }

  @Override
  public Enumeration<URL> getResources(String name) throws IOException {
    /**
     * Similar to super, but local resources are enumerated before parent
     * resources
     */
    Enumeration<URL> systemUrls = null;
    if (system != null) {
      systemUrls = system.getResources(name);
    }

    Enumeration<URL> localUrls = findResources(name);
    Enumeration<URL> parentUrls = null;
    if (getParent() != null) {
      parentUrls = getParent().getResources(name);
    }

    final List<URL> enumUrls = new ArrayList<URL>();
    if (localUrls != null) {
      while (localUrls.hasMoreElements()) {
        URL local = localUrls.nextElement();
        enumUrls.add(local);
      }
    }

    if (systemUrls != null) {
      while (systemUrls.hasMoreElements()) {
        enumUrls.add(systemUrls.nextElement());
      }
    }

    if (parentUrls != null) {
      while (parentUrls.hasMoreElements()) {
        enumUrls.add(parentUrls.nextElement());
      }
    }

    return new Enumeration<URL>() {
      Iterator<URL> iter = enumUrls.iterator();

      public boolean hasMoreElements() {
        return iter.hasNext();
      }

      public URL nextElement() {
        return iter.next();
      }
    };
  }

  @Override
  public InputStream getResourceAsStream(String name) {
    URL url = getResource(name);
    try {
      return url != null ? url.openStream() : null;
    } catch (IOException e) {
    }
    return null;
  }

  @Override
  public int hashCode() {
    final int prime = 31;
    int result = 1;
    result = prime * result + ((parent == null) ? 0 : parent.hashCode());
    result = prime * result + Arrays.hashCode(urls);
    return result;
  }

  @Override
  public boolean equals(Object obj) {
    if (this == obj) {
      return true;
    }
    if (obj == null) {
      return false;
    }
    if (getClass() != obj.getClass()) {
      return false;
    }
    final FunctionClassLoader other = (FunctionClassLoader) obj;
    if (parent == null) {
      if (other.parent != null) {
        return false;
      }
    } else if (!parent.equals(other.parent)) {
      return false;
    }

    if (!Arrays.equals(urls, other.urls)) {
      return false;
    }
    return true;
  }
}
