/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.function;

import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Supplier;

import lombok.RequiredArgsConstructor;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.ProblemSink;

// TODO Support reporting to a metric set - this will give the platform some feedback
@RequiredArgsConstructor
public class ExpensiveResource<T> implements Supplier<T> {

  private final ProblemSink logTo;

  private final String operationName;

  private final Supplier<T> supplier;

  private T built;

  // forks a thread to report run time of the index operation - it assumes the job has stopped once the thread is
  // interrupted
  private Thread startTimer() {
    long startTime = System.currentTimeMillis();
    long warnTimeMillis = 10 * 1000; // 10 seconds
    AtomicBoolean warned = new AtomicBoolean(false);
    Thread thread = new Thread(() -> {
      // keep looping until we get interrupted
      while (!Thread.currentThread().isInterrupted()) {
        long nowMillis = System.currentTimeMillis();
        if (nowMillis > startTime + warnTimeMillis) {
          warned.set(true);
          Problem problem = Problem.info(
              "%s running after %d seconds",
              operationName,
              TimeUnit.SECONDS.convert((nowMillis - startTime), TimeUnit.MILLISECONDS) // millis to seconds
          );
          logTo.log(problem);
        }

        try {
          Thread.sleep(warnTimeMillis);
        } catch (InterruptedException e) {
          break;
        }
      }

      if (warned.get()) {
        long nowMillis = System.currentTimeMillis();
        logTo.log(Problem.info(
            "%s finished in %d seconds",
            operationName,
            TimeUnit.SECONDS.convert(nowMillis - startTime,  TimeUnit.MILLISECONDS)));
      }
    });

    thread.setDaemon(true);
    thread.setName("slow-op[" + operationName + "]");
    thread.start();

    return thread;
  }

  @Override
  public synchronized T get() {
    // this keeps the building in the calling thread, which will allow any thread local stuff to keep working, at the
    // cost of interruptibility (e.g. it'll be harder to interrupt / bypass the operation )
    if (built == null) {
      Thread thread = startTimer();
      try {
        built = supplier.get();
      } finally {
        thread.interrupt();
      }
    }

    return built;
  }
}
