/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.function;

import java.util.List;

import lombok.Getter;
import lombok.RequiredArgsConstructor;
import nz.org.riskscape.engine.rl.RealizableFunction;
import nz.org.riskscape.engine.rl.RealizationContext;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.problem.ProblemException;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.rl.ast.FunctionCall;

/**
 * Base class for DRYing up all the code common to most {@link RealizableFunction}s
 */
@RequiredArgsConstructor
public abstract class BaseRealizableFunction implements RealizableFunction {

  @Getter
  protected final ArgumentList arguments;

  @Getter
  protected final Type returnType;

  /**
   * @return a {@link RiskscapeFunction} that wraps this {@link RealizableFunction}.  Note that I haven't made this
   * class implements {@link RiskscapeFunction} directly as the plan is to re-arrange these classes so that all
   * functions become realizable but are not riskscape functions themselves (until they are realized).
   */
  public RiskscapeFunction asFunction() {
    return RealizableFunction.asFunction(this, arguments, returnType);
  }

  /**
   * Shortcut for asFunction().identified(id)
   */
  public IdentifiedFunction identified(String id) {
    return asFunction().identified(id);
  }

  /**
   * Shortcut for asFunction().builtin(id, category)
   */
  public IdentifiedFunction builtin(String id, IdentifiedFunction.Category category) {
    return asFunction().builtin(id, category);
  }

  /**
   * Implements some basic boilerplate error checks by default. This allows simple
   * realizable functions to just define the {@link #build(RealizationContext, FunctionCall, List)}
   * guts of the function. More complicated realizable functions (i.e. with non-trivial type handling)
   * may want to override this completely.
   */
  @Override
  public ResultOrProblems<RiskscapeFunction> realize(RealizationContext context, FunctionCall functionCall,
      List<Type> givenTypes) {
    return ProblemException.catching(() -> {
      // do the common type error handling up front
      if (!arguments.isCompatible(context, givenTypes)) {
        throw new ProblemException(arguments.getProblems(context, givenTypes));
      }
      return build(context, functionCall, givenTypes);
    });
  }

  /**
   * This method is internal to the sub-class. Simple realizable functions can just implement this, to
   * jump straight to the meat of building the realized function. You should throw a ProblemException
   * for any errors encountered.
   */
  protected RiskscapeFunction build(RealizationContext context, FunctionCall functionCall, List<Type> givenTypes)
      throws ProblemException {
    // this is an optional approach to realizing the function
    throw new UnsupportedOperationException("Not implemented yet");
  }

}
