/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.function;

import java.net.URI;
import java.util.List;

import com.google.common.base.CaseFormat;
import com.google.common.collect.Lists;

import lombok.Getter;
import nz.org.riskscape.engine.types.Type;

/**
 * Convenience class with built in support for integer and floating point modes and everything in between
 */
public abstract class BaseMathsFunction implements IdentifiedFunction {

  @Getter
  private final String id = CaseFormat.UPPER_CAMEL.to(CaseFormat.LOWER_UNDERSCORE, getClass().getSimpleName());
  protected String description = "";

  @Override
  public String getDescription() {
    return description;
  }

  @Override
  public Object call(List<Object> args) {
    throw new RuntimeException("not implemented");
  }

  @Override
  public Category getCategory() {
    return Category.MATHS;
  }

  @Override
  public URI getSourceURI() {
    return RiskscapeFunction.BUILT_IN;
  }

  protected RiskscapeFunction wrapping(Type... newTypes) {
    ArgumentList prevArgs = getArguments();

    if (newTypes.length > prevArgs.size()) {
      throw new IllegalArgumentException("overloading should not declare new args");
    }

    List<FunctionArgument> newArgs = Lists.newArrayList();
    for (int i = 0; i < newTypes.length; i++) {
      newArgs.add(new FunctionArgument(prevArgs.get(i).getKeyword(), newTypes[i]));
    }

    return wrapping(new ArgumentList(newArgs));
  }

  protected RiskscapeFunction wrapping(FunctionArgument... newArgs) {

    return wrapping(ArgumentList.fromArray(newArgs));
  }

  protected RiskscapeFunction wrapping(ArgumentList args) {

    List<Type> argumentTypes = args.getArgumentTypes();

    return new RiskscapeFunction() {

      @Override
      public Object call(List<Object> args) {
        return BaseMathsFunction.this.call(args);
      }

      @Override
      public List<Type> getArgumentTypes() {
        return argumentTypes;
      }

      @Override
      public ArgumentList getArguments() {
        return args;
      }

      @Override
      public Type getReturnType() {
        return BaseMathsFunction.this.getReturnType();
      }
    };
  }

}
