/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.function;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import nz.org.riskscape.engine.ArgsProblems;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.rl.RealizableFunction;
import nz.org.riskscape.engine.rl.RealizationContext;
import nz.org.riskscape.engine.types.FunctionType;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.types.TypeProblems;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.engine.typeset.TypeSet;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.ProblemException;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.rl.ast.FunctionCall;

/**
 * Function for applying arguments to a first-order function, e.g. an {@link UntypedFunction} that is part of a
 * {@link Tuple}
 */
@Deprecated
public class ApplyFunction implements RealizableFunction {

  private final ArgumentList arguments = ArgumentList.create("function", FunctionType.WILD, "args...", Types.ANYTHING);

  @Override
  public ResultOrProblems<RiskscapeFunction> realize(RealizationContext context, FunctionCall functionCall,
      List<Type> argumentTypes) {

    return ProblemException.catching(() -> {
      FunctionType functionType =  Types.findOrThrow("function", FunctionType.WILD, argumentTypes.get(0));

      List<Type> expectedArgs = functionType.getArgumentTypes();
      List<Type> givenFunctionArgs = argumentTypes.subList(1, argumentTypes.size());

      // in the future, we could defer to the function resolving code here to improve adaptability, but for now we just
      // check the arity and that assignment will succeed
      if (expectedArgs.size() != givenFunctionArgs.size()) {
        throw new ProblemException(ArgsProblems.get().wrongNumber(expectedArgs.size() + 1, argumentTypes.size()));
      }
      checkArgVariance(context.getProject().getTypeSet(), expectedArgs.iterator(), givenFunctionArgs.iterator());

      return RiskscapeFunction.create(this, argumentTypes, functionType.getReturnType(), args -> {
        UntypedFunction function = (UntypedFunction) args.get(0);
        return function.call(args.subList(1, args.size()));
      });
    });
  }

  /**
   * Pair-wise comparison of args to make sure the function can be called safely
   */
  private void checkArgVariance(
      TypeSet typeset,
      Iterator<Type> expectedArgs,
      Iterator<Type> givenFunctionArgs
  ) throws ProblemException {
    List<Problem> problems = new ArrayList<>(0);

    int argCounter = 2;
    while (expectedArgs.hasNext()) {
      Type expectedArg = expectedArgs.next();
      Type givenArg = givenFunctionArgs.next();

      if (!typeset.isAssignable(givenArg, expectedArg)) {
        problems.add(TypeProblems.get().mismatch("arg" +  argCounter, expectedArg, givenArg));
      }
      argCounter++;
    }

    ProblemException.throwUnlessEmpty(problems);
  }

  public RiskscapeFunction asFunction() {
    return RealizableFunction.asFunction(this, arguments, Types.ANYTHING);
  }

}
