/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */

package nz.org.riskscape.engine.filter;

import java.util.Collections;
import java.util.List;

import org.geotools.api.filter.Filter;
import org.geotools.api.filter.capability.FunctionName;
import org.geotools.api.filter.expression.Expression;
import org.geotools.api.filter.expression.ExpressionVisitor;
import org.geotools.api.filter.expression.Function;
import org.geotools.api.filter.expression.Literal;

import lombok.AccessLevel;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import nz.org.riskscape.engine.expr.ConstantExpression;
import nz.org.riskscape.engine.expr.TypedExpression;
import nz.org.riskscape.engine.gt.FilterValidator;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.problem.ResultOrProblems;

/**
 * Adaptor that makes a {@link Filter} available as an {@link TypedExpression} that will return
 * a boolean when evaluated.
 */
@RequiredArgsConstructor(access = AccessLevel.PRIVATE)
public class FilterExpression implements Function, TypedExpression {

  public static FilterExpression of(Filter filter) {
    return new FilterExpression(filter);
  }

  /**
   * To defer to when evaluating expression.
   */
  @Getter
  private final Filter filter;

  @Override
  public ResultOrProblems<Type> evaluateType(Type sourceType) {
    return ResultOrProblems.of(Types.BOOLEAN,
        FilterValidator.INSTANCE.validateFilter(sourceType.asStruct(), filter));
  }

  @Override
  public Object evaluate(Object o) {
    return evaluate(o, Types.BOOLEAN.internalType());
  }

  @Override
  @SuppressWarnings("unchecked")
  public <T> T evaluate(Object o, Class<T> type) {
    Object raw = filter.evaluate(o);
    if (! type.isInstance(raw)) {
      return null;
    }
    return (T)raw;
  }

  @Override
  public Object accept(ExpressionVisitor visitor, Object extraData) {
    return visitor.visit(this, extraData);
    // NB support could be added via the Function expression interface, but then to fulfil the interface properly
    // would require that a functionName be registered with the function factory, which would be hard to do here
    // for our purposes, visiting is typically used for rewriting expressions, which we could possibly support with a
    // specific cloning interface.  These two things are not mutually exclusive
    // getParameters could be implemented by visiting the filter and extracting all expressions found in dfs order
  }

  @Override
  public String getName() {
    return filter.toString();
  }

  @Override
  public FunctionName getFunctionName() {
    return null;
  }

  @Override
  public List<Expression> getParameters() {
    return Collections.emptyList();
  }

  @Override
  public Literal getFallbackValue() {
    return new ConstantExpression(false);
  }

}
