/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.expr;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.function.Supplier;

import org.geotools.api.filter.capability.FunctionName;
import org.geotools.api.filter.expression.Expression;
import org.geotools.api.filter.expression.ExpressionVisitor;
import org.geotools.api.filter.expression.Function;
import org.geotools.api.filter.expression.Literal;

import lombok.NonNull;
import nz.org.riskscape.engine.RiskscapeException;
import nz.org.riskscape.engine.UnknownFunctionException;
import nz.org.riskscape.engine.function.FunctionCallException;
import nz.org.riskscape.engine.function.IdentifiedFunction;
import nz.org.riskscape.engine.function.RiskscapeFunction;
import nz.org.riskscape.engine.gt.FilterValidator;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.problem.Problem.Severity;

/**
 * An {@link Expression} which applies a {@link RiskscapeFunction}
 * @deprecated we don't use these expressions any more
 */
// TODO waaaay more validation
@Deprecated
public class RiskscapeFunctionExpression implements Function, TypedExpression {

  private final Supplier<RiskscapeFunction> functionSupplier;
  private final String name;
  private List<? extends Expression> parameters;

  public RiskscapeFunctionExpression(@NonNull String name, @NonNull RiskscapeFunction vf,
      List<? extends Expression> params) {
    this.functionSupplier = () -> vf;
    this.name = name;
    this.parameters = params;
  }

  public RiskscapeFunctionExpression(@NonNull IdentifiedFunction vf, List<? extends Expression> params) {
    this.functionSupplier = () -> vf;
    this.name = vf.getId();
    this.parameters = params;
  }

  public RiskscapeFunctionExpression(@NonNull String name,
      @NonNull Supplier<RiskscapeFunction> vf,
      List<? extends Expression> parameters) {
    this.functionSupplier = vf;
    this.name = name;
    this.parameters = parameters;
  }

  @Override
  public String getName() {
    return name;
  }

  @Override
  public String toString() {
    return String.format("%s(%s)", name, parameters);
  }

  @Override
  public List<Expression> getParameters() {
    return Collections.unmodifiableList(parameters);
  }

  @Override
  public Object evaluate(Object scope) {

    RiskscapeFunction function = functionSupplier.get();
    if (function.getArgumentTypes().size() != parameters.size()) {
      throw new RiskscapeException(String.format("Function %s expects %d arguments but %d were given",
          name, function.getArgumentTypes().size(), parameters.size()));
    }

    List<Object> values = new ArrayList<Object>(parameters.size());
    for (int i = 0; i < parameters.size(); i++) {
      Object paramValue;
      try {
        paramValue = parameters.get(i).evaluate(scope);
      } catch (RuntimeException e) {
        throw new FunctionEvaluationException("Failed to evaluate expression", e);
      }

      Object coerced;
      try {
        coerced = function.getArgumentTypes().get(i).coerce(paramValue);
      } catch (RuntimeException e) {
        throw new FunctionEvaluationException("Failed to coerce argument " + (i + 1), e);
      }
      values.add(coerced);
    }

    try {
      return function.call(values);
    } catch (Exception ex) {
      String message = String.format(
          "Failed to invoke riskscape function %s against %s : %s",
          function,
          values,
          ex);

      throw new FunctionCallException(message, ex);
    }
  }


  @Override
  public ResultOrProblems<Type> evaluateType(Type sourceType) {
    try {
      RiskscapeFunction function = functionSupplier.get();
      if (function.getArgumentTypes().size() != parameters.size()) {
        return ResultOrProblems.failed(new Problem(Severity.ERROR,
            String.format("Function '%s' requires %d arguments but %d were provided",
                name, function.getArgumentTypes().size(), parameters.size())));
      } else {
        List<Problem> problems = new ArrayList<>();
        for (Expression expr : parameters) {
          problems.addAll(FilterValidator.INSTANCE.validateExpression(sourceType, expr).getProblems());
        }
        return ResultOrProblems.of(function.getReturnType(), problems);
      }
    } catch (UnknownFunctionException e) {
      return ResultOrProblems.failed(new Problem(Severity.ERROR, e.getMessage()));
    }
  }

  @Override
  public Object accept(ExpressionVisitor visitor, Object extraData) {
    return visitor.visit(this, extraData);
  }

  @Override
  public FunctionName getFunctionName() {
    return null;
  }
  @Override
  public Literal getFallbackValue() {
    return null;
  }

  @SuppressWarnings("unchecked")
  @Override
  public <T> T evaluate(Object object, Class<T> context) {
    return (T) evaluate(object);
  }


}
