/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.data.relation;

import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.json.simple.parser.JSONParser;
import org.json.simple.parser.ParseException;

import nz.org.riskscape.engine.bind.BindingContext;
import nz.org.riskscape.engine.bind.TypeSafeBinder;
import nz.org.riskscape.engine.problem.ProblemFactory;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.Problems;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.rl.ast.Expression;

public class AttributeMapBinder extends TypeSafeBinder<Object, AttributeMap> {

  public interface LocalProblems extends ProblemFactory {
    Problem couldNotParseJsonString(String errorMessage);
    Problem jsonObjectNotADictionary(String simpleName);
  }

  public static final LocalProblems PROBLEMS = Problems.get(LocalProblems.class);

  public AttributeMapBinder() {
    super(Object.class, AttributeMap.class);
  }

  @Override
  public boolean canBind(Class<?> sourceType, Class<?> destinationType) {
    return super.canBind(sourceType, destinationType)
        && (sourceType.equals(String.class) || Map.class.isAssignableFrom(sourceType));
  }

  @Override
  protected ResultOrProblems<AttributeMap> bindTypedValue(BindingContext context, Object value) {
    Map<?, ?> toBind;
    if (value instanceof String) {
      Object obj;
      try {
        obj = new JSONParser().parse(value.toString());
      } catch (ParseException e) {
        return ResultOrProblems.failed(PROBLEMS.couldNotParseJsonString(e.toString()));
      }

      if (!(obj instanceof Map)) {
        return ResultOrProblems.failed(PROBLEMS.jsonObjectNotADictionary(obj.getClass().getSimpleName()));
      }

      toBind = (Map<?, ?>) obj;
    } else if (value instanceof Map) {
      toBind = (Map<?, ?>) value;
    } else {
      throw new RuntimeException("api misuse - can't bind a " + value.getClass());
    }

    AttributeMap attrMap = new AttributeMap();
    List<Problem> collected = new LinkedList<>();
    for (Map.Entry<?, ?> entry : toBind.entrySet()) {
      String key = entry.getKey().toString();
      ResultOrProblems<Expression> expression = context.bind(entry.getValue(), Expression.class);

      if (expression.isPresent()) {
        attrMap.put(key, expression.get());
      } else {
        collected.add(Problems.foundWith(key, expression.getProblems()));
      }
    }

    if (Problem.hasErrors(collected)) {
      return ResultOrProblems.failed(collected);
    }

    return ResultOrProblems.of(attrMap, collected);
  }


}
