/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */

package nz.org.riskscape.engine.data.coverage;

import org.locationtech.jts.geom.Geometry;

import nz.org.riskscape.engine.SRIDSet;
import nz.org.riskscape.engine.coverage.TypedCoverage;
import nz.org.riskscape.engine.function.ExpensiveResource;
import nz.org.riskscape.engine.geo.IntersectionIndex;
import nz.org.riskscape.engine.query.TupleUtils;
import nz.org.riskscape.engine.query.TupleUtils.FindOption;
import nz.org.riskscape.engine.relation.Relation;
import nz.org.riskscape.engine.relation.SpatialMetadata;
import nz.org.riskscape.engine.relation.TupleIterator;
import nz.org.riskscape.engine.types.Struct.StructMember;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.ProblemSink;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.problem.StandardCodes;
import nz.org.riskscape.engine.types.Type;

/**
 * A {@link TypedCoverage} that is backed by a {@link Relation}. The {@link Type} yielded by the
 * relation must contain a {@link Geometry} attribute.
 */
public class SpatialRelationTypedCoverage extends IndexedTypedCoverage {

  public static ResultOrProblems<SpatialRelationTypedCoverage> of(
      Relation relation,
      SRIDSet sridSet,
      ProblemSink logTo
  ) {
    if (TupleUtils.findGeometryMember(relation.getType(), FindOption.OPTIONAL) == null) {
      return ResultOrProblems.failed(Problem.error(StandardCodes.GEOMETRY_REQUIRED, relation.getType()));
    }

    SpatialRelationTypedCoverage coverage = new SpatialRelationTypedCoverage(relation, sridSet, logTo);
    return ResultOrProblems.of(coverage);
  }

  private final Relation backing;

  private final StructMember geomMember;

  private final ExpensiveResource<IntersectionIndex> indexResource;

  private SpatialRelationTypedCoverage(Relation backing, SRIDSet sridSet, ProblemSink logTo) {
    super(
        backing.getType(),
        sridSet,
        backing.getSpatialMetadata().map(smd -> smd.getCrs()).orElse(DEFAULT_CRS)
    );
    this.backing = backing;

    if (backing.getSpatialMetadata().isPresent()) {
      SpatialMetadata sm = backing.getSpatialMetadata().get();
      this.geomMember = sm.getGeometryStructMember();
    } else {
      this.geomMember = TupleUtils.findRequiredGeometryMember(backing.getType());
    }

    this.indexResource = new ExpensiveResource<>(
        logTo,
        "coverage-index-" + backing.getSourceInformation(),
        () -> {
          return populateIndex();
        }
    );
  }

  @Override
  protected StructMember getGeomMember() {
    return geomMember;
  }

  @Override
  protected IntersectionIndex getIndex() {
    return indexResource.get();
  }

  private IntersectionIndex populateIndex() {
    IntersectionIndex index = IntersectionIndex.withDefaultOptions(geomMember, sridSet);

    try (TupleIterator it = backing.iterator()) {
      while (it.hasNext()) {
        index.insert(it.next());
      }
    }
    index.build();

    return index;
  }

}
