/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.data.coverage;

import java.util.Optional;
import java.util.function.Supplier;

import org.geotools.geometry.jts.ReferencedEnvelope;
import org.locationtech.jts.geom.Coordinate;
import org.locationtech.jts.geom.Point;
import org.geotools.api.referencing.crs.CoordinateReferenceSystem;

import lombok.Getter;
import nz.org.riskscape.engine.SRIDSet;
import nz.org.riskscape.engine.coverage.TypedCoverage;
import nz.org.riskscape.engine.geo.NearestNeighbourIndex;
import nz.org.riskscape.engine.types.Type;

/**
 * A {@link TypedCoverage} that will find the nearest match that is within maxDistance.
 */
public class NearestNeighbourCoverage implements TypedCoverage {

  private NearestNeighbourIndex index;

  private final Supplier<NearestNeighbourIndex> indexSupplier;
  @Getter
  private final Type type;
  @Getter
  private final CoordinateReferenceSystem coordinateReferenceSystem;
  private final SRIDSet sridSet;

  /**
   * @param indexSupplier             to obtain the nearest neighbour index when it is first required.
   * @param type                      the struct type of the tuples that the index will return.
   * @param coordinateReferenceSystem the crs of the indexed geometry. Points will be reprojected to this
   *                                  crs before being used to query the index
   * @param sridSet                   to use for reprojecting geometries
   */
  public NearestNeighbourCoverage(Supplier<NearestNeighbourIndex> indexSupplier, Type type,
      CoordinateReferenceSystem coordinateReferenceSystem, SRIDSet sridSet) {
    this.indexSupplier = indexSupplier;
    this.type = type;
    this.coordinateReferenceSystem = coordinateReferenceSystem;
    this.sridSet = sridSet;
  }

  @Override
  public Object evaluate(Point point) {
    Coordinate reprojected = sridSet.reproject(point, sridSet.get(coordinateReferenceSystem)).getCoordinate();
    return getIndex().query(reprojected);
  }

  @Override
  public Optional<ReferencedEnvelope> getEnvelope() {
    return Optional.of(new ReferencedEnvelope(getIndex().getEnvelope(), coordinateReferenceSystem));
  }

  public NearestNeighbourIndex getIndex() {
    if (index == null) {
      index = indexSupplier.get();
    }
    return index;
  }


}
