/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */

package nz.org.riskscape.engine.data.coverage;

import java.util.List;
import java.util.Optional;
import java.util.function.Function;

import org.geotools.geometry.jts.ReferencedEnvelope;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.Point;
import org.geotools.api.referencing.crs.CoordinateReferenceSystem;

import lombok.Getter;

import nz.org.riskscape.engine.SRIDSet;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.coverage.TypedCoverage;
import nz.org.riskscape.engine.geo.IntersectionIndex;
import nz.org.riskscape.engine.relation.Relation;
import nz.org.riskscape.engine.types.Nullable;
import nz.org.riskscape.engine.types.Struct.StructMember;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.util.Pair;

/**
 * A {@link TypedCoverage} that is backed by a {@link Relation}. The {@link Type} yielded by the
 * relation must contain a {@link Geometry} attribute.
 */
public abstract class IndexedTypedCoverage implements TypedCoverage {

  protected final SRIDSet sridSet;

  @Getter
  protected final Type type;

  @Getter
  protected final CoordinateReferenceSystem coordinateReferenceSystem;

  protected IndexedTypedCoverage(Type type, SRIDSet sridSet, CoordinateReferenceSystem coordinateReferenceSystem) {
    // The result type is nullable, because there may not be an interesection
    this.type = Nullable.of(type);
    this.sridSet = sridSet;
    this.coordinateReferenceSystem = coordinateReferenceSystem;
  }

  protected abstract StructMember getGeomMember();
  protected abstract IntersectionIndex getIndex();

  @Override
  public Object evaluate(Point point) {
    IntersectionIndex index = getIndex();
    List<Tuple> intersections = index.findPointIntersections(point);

    // We return the first match that intersects with the point. Technically there could be more
    // than one, should features overlap (or be stacked), or more likely if the point is on the
    // boundary of features.
    // NB in the future, we could 'interpolate' multiple results using an aggregate function, but that seems
    // overkill rn
    return intersections.isEmpty() ? null : intersections.get(0);
  }

  @Override
  public Optional<Function<Geometry, List<Pair<Geometry, Object>>>> getEvaluateIntersectionOp() {
    return Optional.of(geom -> evaluateIntersection(geom));
  }

  // generics removed - because it makes my thick head spin, and I suspect the answer would mean touching a lot of files
  // to futz it and fix it.  With the generics in place, the compiler doesn't like the assignment of
  // List<Pair<Geometry, Tuple>> to List<Pair<Geometry, Object>> - I suspect adding a type variable to TypedCoverage
  // would fix it, but that'll probably cause more problems elsewhere.  I'm 99% sure the assignment is safe, so I'm
  // going to remove the generics
  @SuppressWarnings({ "rawtypes", "unchecked" })
  List<Pair<Geometry, Object>> evaluateIntersection(Geometry geom) {
    List list = getIndex().findIntersections(geom);
    return list;
  }

  @Override
  public Optional<ReferencedEnvelope> getEnvelope() {
    return Optional.of(getIndex().getReferencedEnvelope());
  }
}
