/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.data.coverage;

import java.util.List;

import lombok.AllArgsConstructor;
import lombok.Getter;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.types.Types;

/**
 * Wrapper that maps a two-dimensional (x, y) grid to an underlying set of data.
 * This allows us to:
 * - easily vary the underlying data we're storing (via subclassing),
 * e.g. in DatasetCoverage we need to support both doubles and lists of doubles.
 * - actually use a one-dimensional array underneath (for memory efficiency).
 * - potentially calculate the underlying data on demand, e.g. for huge HDF5
 * datasets that we can't fit in memory in a single array.
 */
@AllArgsConstructor
public abstract class Dataset2D {

  /**
   * Maps a simple one-dimensional array of doubles to a two-dimensional grid, with the dataset yielding the double
   * at each grid position.
   *
   * Data should be organized in the data array to be `x, y`, that is the first `width` number of values are for `y = 0`
   */
  public static Dataset2D fromDoubles(int width, int height, double[] data) {
    return new Dataset2DDoubles(width, height, data);
  }

  /**
   * Combines multiple datasets together and returns a combined struct each time
   * they are sampled.
   */
  public static Dataset2D combine(List<Dataset2D> datasets, Struct combinedType) {
    return new CombinedDataset2D(datasets, combinedType);
  }

  static final class Dataset2DDoubles extends Dataset2D {

    private final double[] data;

    Dataset2DDoubles(long width, long height, double[] data) {
      super(width, height);
      this.data = data;
    }

    @Override
    public Double getValue(long x, long y) {
      int index = getIndex(x, y);
      return data[index];
    }

    @Override
    public Type getType() {
      return Types.FLOATING;
    }
  }

  static final class CombinedDataset2D extends Dataset2D {

    private final List<Dataset2D> datasets;

    @Getter
    private final Struct type;

    CombinedDataset2D(List<Dataset2D> datasets, Struct combinedType) {
      super(datasets.get(0).getWidth(), datasets.get(0).getHeight());
      this.datasets = datasets;
      this.type = combinedType;
    }

    @Override
    public Tuple getValue(long x, long y) {
      // TODO it might be more efficient to build all the Tuples up front once, rather
      // than creating a new one each time the dataset is sampled
      Object[] values = new Object[datasets.size()];
      for (int i = 0; i < datasets.size(); i++) {
        values[i] = datasets.get(i).getValue(x, y);
      }
      return Tuple.ofValues(type, values);
    }
  }

  @Getter
  private long width; /* x dimension */
  @Getter
  private long height; /* y dimension */

  /*
   * helper to convert x, y co-ordinates to a one-dimensional array index
   */
  protected int getIndex(long x, long y) {
    long index = y * width + x;
    if (index > Integer.MAX_VALUE) {
      throw new IllegalStateException("Dataset co-ordinates too big for array index");
    }
    return (int) index;
  }

  /**
   * @return whether the x, y co-ordinates fall within this dataset grid
   */
  public boolean isInBounds(long x, long y) {
    boolean xInBounds = (0 <= x && x < width);
    boolean yInBounds = (0 <= y && y < height);
    return xInBounds && yInBounds;
  }

  /**
   * @return the data stored at the given x, y co-ordinates.
   * Warning: you should verify the co-ordinates via isInBounds() first
   */
  public abstract Object getValue(long x, long y);

  /**
   * @return the type of data stored by the dataset.
   * Similar to TypedCoverage.getType()
   */
  public abstract Type getType();
}
