/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.data.coverage;

import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.Point;
import nz.org.riskscape.engine.SRIDSet;
import nz.org.riskscape.engine.coverage.TypedCoverage;
import org.geotools.geometry.Position2D;
import org.geotools.geometry.jts.ReferencedEnvelope;
import org.geotools.api.geometry.Position;

import lombok.Getter;

/**
 * Provides utility functions useful to {@link TypedCoverage} implementations.
 */
public class CoverageValueProviderHelper {

  /**
   * Exception to throw if the helper is given an empty geometry. Checked to ensure calling code handles
   * this correctly.
   */
  public class EmptyGeometryException extends Exception {}

  @Getter
  private final ReferencedEnvelope envelope;
  private final SRIDSet sridSet;
  private final int gridSrid;

  /**
   * Get a helper for a coverage that covers the envelope.
   * @param envelope containing the coverage
   * @param sridSet
   */
  public CoverageValueProviderHelper(ReferencedEnvelope envelope, SRIDSet sridSet) {
    this.envelope = envelope;
    this.sridSet = sridSet;
    // For a lot of sampling operations, getting the srid from a crs adds up to quite a lot (16% of execution time,
    // apparently.  Doing this once saves a lot of wasted CPU cycles)
    this.gridSrid = sridSet.get(envelope.getCoordinateReferenceSystem());
  }

  /**
   * Will reproject geom into the {@link org.geotools.api.referencing.crs.CoordinateReferenceSystem} of the coverage if
   * necessary.
   * @param geom
   * @return geom, or reproject geom if necessary
   * @throws EmptyGeometryException if geom is empty
   */
  public Geometry reprojectIfNecessary(Geometry geom) throws EmptyGeometryException {
    if (geom.isEmpty()) {
      // you can't reproject an empty geometry
      throw new EmptyGeometryException();
    }
    if (geom.getSRID() != gridSrid) {
      return sridSet.reproject(geom, gridSrid);
    }
    return geom;
  }

  /**
   * @param position
   * @return true if position is within the envelope
   */
  public boolean contains(Position position) {
    return envelope.contains(position);
  }

  public Position pointToDirectPosition(Point point) {
    return new Position2D(envelope.getCoordinateReferenceSystem(), point.getX(), point.getY());
  }
}
