/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */

package nz.org.riskscape.engine.data;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import com.google.common.base.Strings;

import nz.org.riskscape.engine.IdentifiedCollection;
import nz.org.riskscape.engine.bind.BindingContext;
import nz.org.riskscape.engine.bind.Parameter;
import nz.org.riskscape.engine.output.Format;
import nz.org.riskscape.engine.problem.GeneralProblems;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.Problems;
import nz.org.riskscape.problem.ResultOrProblems;

public class DefaultBookmarkResolvers extends IdentifiedCollection.Base<BookmarkResolver> implements BookmarkResolvers {

  @Override
  public ResultOrProblems<ResolvedBookmark> resolve(Bookmark bookmark, BindingContext context) {
    List<Problem> problems = new ArrayList<>();

    // if the bookmark has no location, then no resolver will be able to open it
    if (bookmark.getLocation() == null) {
      return ResultOrProblems.failed(GeneralProblems.required("location", Parameter.class));
    }

    for (BookmarkResolver resolver : this.getAll()) {
      Optional<ResolvedBookmark> resolved;
      try {
        resolved = resolver.resolve(bookmark, context);
      } catch (IOException e) {
        problems.add(Problem.warning(e,"I/O error with resolver %s - %s", resolver, e.getMessage()));
        continue;
      }

      if (resolved.isPresent()) {
        return ResultOrProblems.of(resolved.get(), resolved.get().validate());
      }
    }

    if (bookmark.isFromURI()) {
      // the id will either be the URI or some supplied text
      problems.add(0, BookmarkProblems.get().notBookmarkOrFile(bookmark.getId()));
    } else if (Strings.isNullOrEmpty(bookmark.getFormat())) {
      // cannot infer format from the location, user needs to specify it in the bookmark
      problems.add(0, BookmarkProblems.get().couldNotResolve(bookmark.getLocation().toString()));
    } else {
      // format specified, but still couldn't open it
      problems.add(0, GeneralProblems.get().noSuchObjectExistsDidYouMean(
          bookmark.getFormat(), Format.class, getSupportedFormats()).withChildren(
              BookmarkProblems.get().unknownFormatTip()
        ));
    }

    return ResultOrProblems.failed(problems);
  }

  @Override
  public ResultOrProblems<ResolvedBookmark> resolveAndValidate(
      Bookmark bookmark, BindingContext context, Class<?> requiredType) {
    ResultOrProblems<ResolvedBookmark> resolvedOr = resolve(bookmark, context);
    ResolvedBookmark resolved;
    if (resolvedOr.hasErrors()) {
      return resolvedOr.composeFailure(Problems.foundWith(bookmark));
    } else {
      resolved = resolvedOr
          // throw any warnings away for now. they won't get lost because they still exist in
          // resolvedOr which will get returned. but we drain them here so they don't get in the way
          // of the validation, which may result in some more serious problems.
          .getWithProblemsIgnored();
    }

    if (resolved.hasValidationErrors()) {
      return ResultOrProblems.failed(Problems.foundWith(bookmark, resolved.validate()));
    } else {

      ResultOrProblems<?> dataOr = resolved.getData(requiredType);
      if (dataOr.hasErrors()) {
        return dataOr.composeFailure(Problems.foundWith(bookmark));
      }

      return resolvedOr;
    }
  }

  @Override
  public <T> ResultOrProblems<T> getData(
      Bookmark bookmark,
      BindingContext context,
      Class<T> requiredType
  ) {
    return resolveAndValidate(bookmark, context, requiredType).map(rb -> rb.getData(requiredType).get());
  }


}
