/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.bind.impl;

import java.util.Arrays;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import nz.org.riskscape.engine.bind.BindingContext;
import nz.org.riskscape.engine.bind.CompatBinder;
import nz.org.riskscape.engine.bind.Parameter;
import nz.org.riskscape.engine.bind.ParameterBindingException;
import nz.org.riskscape.engine.sort.SortBy;
import nz.org.riskscape.engine.sort.SortBy.Direction;
import nz.org.riskscape.rl.ast.Expression;

/**
 * Converts a string to an {@link SortBy}. The string is expected to be of the format:
 *
 * <RL Expression>( [ASC|DESC])
 *
 * If the ordering(ASC|DESC) is not specified then default ASC is used.
 */

public class SortByBinder extends CompatBinder {

  private static final Pattern NOT_A_COMPILER = Pattern.compile("^([^\\s]+)(?:\\s+(.+))?$");
  public static final Direction DEFAULT_DIRECTION = Direction.ASC;


  @Override
  public boolean canBind(Parameter modelParameter) {
    return SortBy.class.isAssignableFrom(modelParameter.getType());
  }

  @Override
  public Object bind(BindingContext context, Parameter parameter, String sortByExpression) {
    sortByExpression = sortByExpression.trim();

    Matcher m = NOT_A_COMPILER.matcher(sortByExpression);
    if (m.find()) {
      String expressionString = m.group(1);
      Expression expression = parseExpression(context, parameter, expressionString);
      Direction direction = parseDirection(parameter, m.group(2));

      return new SortBy(expression, direction);
    } else {
      throw new ParameterBindingException(parameter, "Could not parse sorting expression '%s'", sortByExpression);
    }
  }

  private Expression parseExpression(BindingContext context, Parameter parameter, String toParse) {
    return context.bind(toParse, Expression.class)
      .orElseThrow(probs ->
        new ParameterBindingException(parameter, probs)
      );
  }

  private Direction parseDirection(Parameter parameter, String toParse) {
    if (toParse == null) {
      return DEFAULT_DIRECTION;
    } else {
      try {
        return Direction.valueOf(toParse.toUpperCase());
      } catch (IllegalArgumentException ex) {
        throw new ParameterBindingException(
            parameter,
            "Bad direction '%s', should be one of %s",
            toParse,
            Arrays.asList(Direction.values()));
      }
    }
  }
}
