/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.bind.impl;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import nz.org.riskscape.engine.bind.BaseBinder;
import nz.org.riskscape.engine.bind.BindingContext;
import nz.org.riskscape.engine.problem.GeneralProblems;
import nz.org.riskscape.engine.problem.ProblemFactory;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.Problems;
import nz.org.riskscape.problem.ResultOrProblems;

public class NumberBinder extends BaseBinder {

  public interface LocalProblems extends ProblemFactory {
    Problem numberFormatException(String value, Class<?> numberType);
  }

  public static final LocalProblems PROBLEMS = Problems.get(LocalProblems.class);

  /**
   * The types of {@link Number} that are acceptable as destination types.
   *
   * All of the number types in this list should have a constructor that accepts a single string argument.
   */
  public static final List<Class> SUPPORTED_NUMBER_TYPES = Arrays.asList(
      Integer.class, Long.class, Float.class, Double.class, BigDecimal.class);

  @Override
  public boolean canBind(Class<?> sourceType, Class<?> destinationType) {
    return SUPPORTED_NUMBER_TYPES.contains(destinationType)
        && (sourceType.equals(String.class) || Number.class.isAssignableFrom(sourceType));
  }

  @Override
  public <T> ResultOrProblems<T> bindValue(BindingContext context, Object valueObj, Class<T> destinationType) {
    return bindValue(valueObj, destinationType);
  }

  public <T> ResultOrProblems<T> bindValue(Object valueObj, Class<T> destinationType) {
    if (valueObj instanceof String) {
      String value = valueObj.toString();  // tostring should be good enough for basic number conversion to work
      try {
        Constructor<?> constructor = destinationType.getConstructor(String.class);
        Object result = constructor.newInstance(value);
        return ResultOrProblems.of(destinationType.cast(result));
      } catch (SecurityException | ReflectiveOperationException | IllegalArgumentException e) {
        if (e instanceof InvocationTargetException && e.getCause() instanceof NumberFormatException) {
          return ResultOrProblems.failed(PROBLEMS.numberFormatException(value, destinationType));
        } else {
          throw new RuntimeException("Reflection operation failed unexpectedly", e);
        }
      }
    }

    // valueObj must be a number
    Number numberValue = (Number)valueObj;
    Number result;
    if (destinationType.equals(Integer.class)) {
      result = numberValue.intValue();
    } else if (destinationType.equals(Long.class)) {
      result = numberValue.longValue();
    } else if (destinationType.equals(Float.class)) {
      result = numberValue.floatValue();
    } else if (destinationType.equals(Double.class)) {
      result = numberValue.doubleValue();
    } else {
      result = new BigDecimal(numberValue.toString());
    }
    List<Problem> warnings = new ArrayList<>();
    if (! areEquivalent(numberValue, result)) {
      warnings.add(GeneralProblems.get().precisionLoss(numberValue, result));
    }
    return ResultOrProblems.of(destinationType.cast(result), warnings);
  }

  private boolean areEquivalent(Number n1, Number n2) {
    BigDecimal d1 = new BigDecimal(n1.toString());
    BigDecimal d2 = new BigDecimal(n2.toString());
    return d1.compareTo(d2) == 0;
  }

}
